"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.retryIfBulkOperationConflicts = void 0;
var _pMap = _interopRequireDefault(require("p-map"));
var _lodash = require("lodash");
var _apmUtils = require("@kbn/apm-utils");
var _lib = require("../../lib");
var _wait_before_next_retry = require("./wait_before_next_retry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_RULES_IDS_IN_RETRY = 1000;
const retryIfBulkOperationConflicts = async ({
  action,
  logger,
  bulkOperation,
  filter,
  retries = _wait_before_next_retry.RETRY_IF_CONFLICTS_ATTEMPTS
}) => {
  return (0, _apmUtils.withSpan)({
    name: 'retryIfBulkOperationConflicts',
    type: 'rules'
  }, () => handler({
    action,
    logger,
    bulkOperation,
    filter,
    retries
  }));
};
exports.retryIfBulkOperationConflicts = retryIfBulkOperationConflicts;
const handler = async ({
  action,
  logger,
  bulkOperation,
  filter,
  accListSpecificForBulkOperation,
  // this list can include several accumulators, depends on the type of bulk operation
  accErrors = [],
  accRules = [],
  retries = _wait_before_next_retry.RETRY_IF_CONFLICTS_ATTEMPTS
}) => {
  try {
    const {
      errors: currentErrors,
      rules: currentRules,
      accListSpecificForBulkOperation: currentAccListSpecificForBulkOperation
    } = await bulkOperation(filter);

    // it a way to assign default value for a accListSpecificForBulkOperation
    if (!accListSpecificForBulkOperation) {
      accListSpecificForBulkOperation = Array.from(Array(currentAccListSpecificForBulkOperation.length), () => []);
    }
    const rules = [...accRules, ...currentRules];
    const errors = retries <= 0 ? [...accErrors, ...currentErrors] : [...accErrors, ...currentErrors.filter(error => error.status !== 409)];
    const ruleIdsWithConflictError = currentErrors.reduce((acc, error) => {
      if (error.status === 409) {
        return [...acc, error.rule.id];
      }
      return acc;
    }, []);
    const newAccListSpecificForBulkOperation = accListSpecificForBulkOperation.map((acc, index) => [...acc, ...currentAccListSpecificForBulkOperation[index]]);
    if (ruleIdsWithConflictError.length === 0) {
      return {
        errors,
        rules,
        accListSpecificForBulkOperation: newAccListSpecificForBulkOperation
      };
    }
    if (retries <= 0) {
      logger.warn(`Bulk ${action.toLowerCase()} rules conflicts, exceeded retries`);
      return {
        errors,
        rules,
        accListSpecificForBulkOperation: newAccListSpecificForBulkOperation
      };
    }
    logger.debug(`Bulk ${action.toLowerCase()} rules conflicts, retrying ..., ${ruleIdsWithConflictError.length} saved objects conflicted`);
    await (0, _wait_before_next_retry.waitBeforeNextRetry)(retries);

    // here, we construct filter query with ids. But, due to a fact that number of conflicted saved objects can exceed few thousands we can encounter following error:
    // "all shards failed: search_phase_execution_exception: [query_shard_exception] Reason: failed to create query: maxClauseCount is set to 2621"
    // That's why we chunk processing ids into pieces by size equals to MAX_RULES_IDS_IN_RETRY
    return (await (0, _pMap.default)((0, _lodash.chunk)(ruleIdsWithConflictError, MAX_RULES_IDS_IN_RETRY), async queryIds => handler({
      action,
      logger,
      bulkOperation,
      filter: (0, _lib.convertRuleIdsToKueryNode)(queryIds),
      accListSpecificForBulkOperation: newAccListSpecificForBulkOperation,
      accErrors: errors,
      accRules: rules,
      retries: retries - 1
    }), {
      concurrency: 1
    })).reduce((acc, item) => {
      return {
        errors: [...acc.errors, ...item.errors],
        rules: [...acc.rules, ...item.rules],
        accListSpecificForBulkOperation: acc.accListSpecificForBulkOperation.map((element, index) => [...element, ...item.accListSpecificForBulkOperation[index]])
      };
    }, {
      errors: [],
      rules: [],
      accListSpecificForBulkOperation: Array.from(Array(accListSpecificForBulkOperation.length), () => [])
    });
  } catch (err) {
    throw err;
  }
};