"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.schemaToIoTs = schemaToIoTs;
var tslib_1 = require("tslib");
var t = tslib_1.__importStar(require("io-ts"));
var excess_1 = require("./excess");
/**
 * Is it a tuple of t.Mixed?
 * @param schemas Array of io-ts schemas
 */
function isOneOfCandidate(schemas) {
    return schemas.length === 2;
}
/**
 * Converts each {@link SchemaValue} to the io-ts equivalent
 * @param value The {@link SchemaValue} to parse
 */
function schemaValueToIoTs(value) {
    // We need to check the pass_through type on top of everything
    if (value.type === 'pass_through') {
        return t.unknown;
    }
    if ('properties' in value) {
        var _a = value.properties, DYNAMIC_KEY = _a.DYNAMIC_KEY, properties = tslib_1.__rest(_a, ["DYNAMIC_KEY"]);
        var schemas = [schemaObjectToIoTs({ properties: properties })];
        if (DYNAMIC_KEY) {
            schemas.push(t.record(t.string, schemaValueToIoTs(DYNAMIC_KEY)));
        }
        return isOneOfCandidate(schemas) ? t.union(schemas) : schemas[0];
    }
    else {
        var valueType = value.type; // Copied in here because of TS reasons, it's not available in the `default` case
        switch (valueType) {
            case 'boolean':
                return t.boolean;
            case 'keyword':
            case 'text':
            case 'date':
                return t.string;
            case 'byte':
            case 'double':
            case 'float':
            case 'integer':
            case 'long':
            case 'short':
                return t.number;
            case 'array':
                if ('items' in value) {
                    return t.array(schemaValueToIoTs(value.items));
                }
                throw new Error("Schema type must include the \"items\" declaration.");
            default:
                throw new Error("Unsupported schema type ".concat(valueType, "."));
        }
    }
}
/**
 * Loops through a list of [key, SchemaValue] tuples to convert them into a valid io-ts parameter to define objects.
 * @param entries Array of tuples [key, {@link SchemaValue}]. Typically, coming from Object.entries(SchemaObject).
 */
function entriesToObjectIoTs(entries) {
    return Object.fromEntries(entries.map(function (_a) {
        var _b = tslib_1.__read(_a, 2), key = _b[0], value = _b[1];
        try {
            return [key, schemaValueToIoTs(value)];
        }
        catch (err) {
            err.failedKey = tslib_1.__spreadArray([key], tslib_1.__read((err.failedKey || [])), false);
            throw err;
        }
    }));
}
/**
 * Converts a {@link SchemaObject} to the io-ts equivalent.
 * @param schemaObject The {@link SchemaObject} to parse.
 */
function schemaObjectToIoTs(schemaObject) {
    var objectEntries = Object.entries(schemaObject.properties);
    var requiredFields = objectEntries.filter(function (_a) {
        var _b = tslib_1.__read(_a, 2), key = _b[0], _meta = _b[1]._meta;
        return (_meta === null || _meta === void 0 ? void 0 : _meta.optional) !== true;
    });
    var optionalFields = objectEntries.filter(function (_a) {
        var _b = tslib_1.__read(_a, 2), key = _b[0], _meta = _b[1]._meta;
        return (_meta === null || _meta === void 0 ? void 0 : _meta.optional) === true;
    });
    return (0, excess_1.excess)(t.intersection([t.interface(entriesToObjectIoTs(requiredFields)), t.partial(entriesToObjectIoTs(optionalFields))]));
}
/**
 * Converts a {@link RootSchema} to an io-ts validation object.
 * @param rootSchema The {@link RootSchema} to be parsed.
 */
function schemaToIoTs(rootSchema) {
    try {
        return schemaObjectToIoTs({ properties: rootSchema });
    }
    catch (err) {
        if (err.failedKey) {
            err.message = "Malformed schema for key [".concat(err.failedKey.join('.'), "]: ").concat(err.message);
        }
        throw err;
    }
}
