"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getHealth = exports.getAlertingHealthStatus = void 0;
var _alertingTypes = require("@kbn/alerting-types");
var _saved_objects = require("../saved_objects");
var _types = require("../types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getHealth = async internalSavedObjectsRepository => {
  const healthStatuses = {
    decryptionHealth: {
      status: _alertingTypes.HealthStatus.OK,
      timestamp: ''
    },
    executionHealth: {
      status: _alertingTypes.HealthStatus.OK,
      timestamp: ''
    },
    readHealth: {
      status: _alertingTypes.HealthStatus.OK,
      timestamp: ''
    }
  };
  const {
    saved_objects: decryptErrorData
  } = await internalSavedObjectsRepository.find({
    filter: `alert.attributes.executionStatus.status:error and alert.attributes.executionStatus.error.reason:${_types.RuleExecutionStatusErrorReasons.Decrypt}`,
    fields: ['executionStatus'],
    type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
    sortField: 'executionStatus.lastExecutionDate',
    sortOrder: 'desc',
    page: 1,
    perPage: 1,
    namespaces: ['*']
  });
  if (decryptErrorData.length > 0) {
    healthStatuses.decryptionHealth = {
      status: _alertingTypes.HealthStatus.Warning,
      timestamp: decryptErrorData[0].attributes.executionStatus.lastExecutionDate
    };
  }
  const {
    saved_objects: executeErrorData
  } = await internalSavedObjectsRepository.find({
    filter: `alert.attributes.executionStatus.status:error and alert.attributes.executionStatus.error.reason:${_types.RuleExecutionStatusErrorReasons.Execute}`,
    fields: ['executionStatus'],
    type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
    sortField: 'executionStatus.lastExecutionDate',
    sortOrder: 'desc',
    page: 1,
    perPage: 1,
    namespaces: ['*']
  });
  if (executeErrorData.length > 0) {
    healthStatuses.executionHealth = {
      status: _alertingTypes.HealthStatus.Warning,
      timestamp: executeErrorData[0].attributes.executionStatus.lastExecutionDate
    };
  }
  const {
    saved_objects: readErrorData
  } = await internalSavedObjectsRepository.find({
    filter: `alert.attributes.executionStatus.status:error and alert.attributes.executionStatus.error.reason:${_types.RuleExecutionStatusErrorReasons.Read}`,
    fields: ['executionStatus'],
    type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
    sortField: 'executionStatus.lastExecutionDate',
    sortOrder: 'desc',
    page: 1,
    perPage: 1,
    namespaces: ['*']
  });
  if (readErrorData.length > 0) {
    healthStatuses.readHealth = {
      status: _alertingTypes.HealthStatus.Warning,
      timestamp: readErrorData[0].attributes.executionStatus.lastExecutionDate
    };
  }
  const {
    saved_objects: noErrorData
  } = await internalSavedObjectsRepository.find({
    filter: 'not alert.attributes.executionStatus.status:error',
    fields: ['executionStatus'],
    type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
    sortField: 'executionStatus.lastExecutionDate',
    sortOrder: 'desc',
    namespaces: ['*']
  });
  const lastExecutionDate = noErrorData.length > 0 ? noErrorData[0].attributes.executionStatus.lastExecutionDate : new Date().toISOString();
  for (const [, statusItem] of Object.entries(healthStatuses)) {
    if (statusItem.status === _alertingTypes.HealthStatus.OK) {
      statusItem.timestamp = lastExecutionDate;
    }
  }
  return healthStatuses;
};
exports.getHealth = getHealth;
const getAlertingHealthStatus = async (savedObjects, stateRuns) => {
  const alertingHealthStatus = await getHealth(savedObjects.createInternalRepository([_saved_objects.RULE_SAVED_OBJECT_TYPE]));
  const state = {
    runs: stateRuns + 1,
    health_status: alertingHealthStatus.decryptionHealth.status
  };
  return {
    state
  };
};
exports.getAlertingHealthStatus = getAlertingHealthStatus;