"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.handleAgentConfigurationSearch = handleAgentConfigurationSearch;
var _search_configurations = require("./search_configurations");
var _mark_error = require("./mark_error");
var _mark_applied_by_agent = require("./mark_applied_by_agent");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function handleAgentConfigurationSearch({
  params,
  internalESClient,
  logger
}) {
  const {
    service,
    etag,
    mark_as_applied_by_agent: markAsAppliedByAgent,
    error
  } = params;
  const configuration = await (0, _search_configurations.searchConfigurations)({
    service,
    internalESClient
  });
  if (!configuration) {
    logger.debug(`[Central configuration] Config was not found for ${service.name}/${service.environment}`);
    return null;
  }
  if (error !== null && error !== void 0 && error.trim()) {
    await (0, _mark_error.markError)({
      id: configuration._id,
      body: configuration._source,
      error,
      internalESClient
    });
    return configuration;
  }

  // whether to update `applied_by_agent` field
  // It will be set to true of the etags match or if `markAsAppliedByAgent=true`
  // `markAsAppliedByAgent=true` means "force setting it to true regardless of etag". This is needed for Jaeger agent that doesn't have etags
  const willMarkAsApplied = (markAsAppliedByAgent || etag === configuration._source.etag) && !configuration._source.applied_by_agent;
  logger.debug(`[Central configuration] Config was found for:
        service.name = ${service.name},
        service.environment = ${service.environment},
        etag (requested) = ${etag},
        etag (existing) = ${configuration._source.etag},
        markAsAppliedByAgent = ${markAsAppliedByAgent},
        willMarkAsApplied = ${willMarkAsApplied}`);
  if (willMarkAsApplied) {
    await (0, _mark_applied_by_agent.markAppliedByAgent)({
      id: configuration._id,
      body: configuration._source,
      internalESClient
    });
  }
  return configuration;
}