"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createHandlers = exports.createDispatchedHandlerFactory = exports.createBaseHandlers = exports.assignHandlers = void 0;
var _lodash = require("lodash");
var _embeddable = require("../state/actions/embeddable");
var _canvas_filters_service = require("../services/canvas_filters_service");
var _resolved_args = require("../state/actions/resolved_args");
var _elements = require("../state/actions/elements");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// @ts-expect-error unconverted file

// This class creates stub handlers to ensure every element and renderer fulfills the contract.
// TODO: consider warning if these methods are invoked but not implemented by the renderer...?

// We need to move towards only using these handlers and ditching our canvas specific ones
const createBaseHandlers = () => ({
  done() {},
  reload() {},
  update() {},
  event() {},
  onDestroy() {},
  getRenderMode: () => 'view',
  isSyncColorsEnabled: () => false,
  isSyncTooltipsEnabled: () => false,
  isSyncCursorEnabled: () => true,
  isInteractive: () => true,
  getExecutionContext: () => undefined
});
exports.createBaseHandlers = createBaseHandlers;
const createHandlers = (baseHandlers = createBaseHandlers()) => ({
  ...baseHandlers,
  destroy() {},
  getElementId() {
    return '';
  },
  getFilter() {
    return '';
  },
  onComplete(fn) {
    this.done = fn;
  },
  onDestroy(fn) {
    this.destroy = fn;
  },
  // TODO: these functions do not match the `onXYZ` and `xyz` pattern elsewhere.
  onEmbeddableDestroyed() {},
  onEmbeddableInputChange() {},
  onResize(fn) {
    this.resize = fn;
  },
  resize(_size) {}
});
exports.createHandlers = createHandlers;
const assignHandlers = (handlers = {}) => Object.assign(createHandlers(), handlers);

// TODO: this is a legacy approach we should unravel in the near future.
exports.assignHandlers = assignHandlers;
const createDispatchedHandlerFactory = dispatch => {
  let isComplete = false;
  let oldElement;
  let completeFn = () => {};
  return element => {
    // reset isComplete when element changes
    if (!(0, _lodash.isEqual)(oldElement, element)) {
      isComplete = false;
      oldElement = element;
    }
    const filters = (0, _canvas_filters_service.getCanvasFiltersService)();
    const handlers = {
      ...createHandlers(),
      event(event) {
        switch (event.name) {
          case 'embeddableInputChange':
            this.onEmbeddableInputChange(event.data);
            break;
          case 'applyFilterAction':
            filters.updateFilter(element.id, event.data);
            dispatch((0, _elements.fetchAllRenderables)());
            break;
          case 'onComplete':
            this.onComplete(event.data);
            break;
          case 'embeddableDestroyed':
            this.onEmbeddableDestroyed();
            break;
          case 'resize':
            this.resize(event.data);
            break;
          case 'onResize':
            this.onResize(event.data);
            break;
        }
      },
      getFilter() {
        return element.filter || '';
      },
      onComplete(fn) {
        completeFn = fn;
      },
      getElementId: () => element.id,
      onEmbeddableInputChange(embeddableExpression) {
        dispatch((0, _embeddable.updateEmbeddableExpression)({
          elementId: element.id,
          embeddableExpression
        }));
      },
      onEmbeddableDestroyed() {
        const argumentPath = [element.id, 'expressionRenderable'];
        dispatch((0, _resolved_args.clearValue)({
          path: argumentPath
        }));
        dispatch((0, _embeddable.fetchEmbeddableRenderable)(element.id));
      },
      done() {
        // don't emit if the element is already done
        if (isComplete) {
          return;
        }
        isComplete = true;
        completeFn();
      }
    };
    return handlers;
  };
};
exports.createDispatchedHandlerFactory = createDispatchedHandlerFactory;