"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processObservable = exports.getObservablesFromEcs = exports.getIPType = exports.getHashFields = void 0;
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _observables = require("../../../common/constants/observables");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getIPType = ip => {
  if (ip.includes(':')) {
    return 'IPV6';
  }
  return 'IPV4';
};

// https://www.elastic.co/docs/reference/ecs/ecs-hash
exports.getIPType = getIPType;
const HASH_FIELDS = ['cdhash', 'md5', 'sha1', 'sha256', 'sha384', 'sha512', 'ssdeep', 'tlsh'];

// https://www.elastic.co/docs/reference/ecs/ecs-hash
const HASH_PARENTS = ['dll', 'file', 'process'];
const getHashFields = () => HASH_PARENTS.map(parent => HASH_FIELDS.map(field => `${parent}.hash.${field}`)).flat();
exports.getHashFields = getHashFields;
const processObservable = (observablesMap, value, typeKey, description) => {
  const key = `${typeKey}-${value}`;
  if (observablesMap.has(key)) {
    return;
  }
  observablesMap.set(key, {
    typeKey,
    value,
    description
  });
};

// helper function to get observables from array of flattened ECS data
exports.processObservable = processObservable;
const getObservablesFromEcs = ecsDataArray => {
  const observablesMap = new Map();
  const description = _i18n.i18n.translate('xpack.cases.caseView.observables.autoExtract.description', {
    defaultMessage: 'Auto extracted observable'
  });
  const hashFields = getHashFields();
  for (const ecsData of ecsDataArray) {
    for (const datum of ecsData) {
      if (datum.value) {
        // Source IP
        if (datum.field === 'source.ip') {
          const ips = (0, _lodash.castArray)(datum.value);
          ips.forEach(ip => {
            const ipType = getIPType(ip);
            processObservable(observablesMap, ip, ipType === 'IPV4' ? _observables.OBSERVABLE_TYPE_IPV4.key : _observables.OBSERVABLE_TYPE_IPV6.key, description);
          });
        }

        // Destination IP
        if (datum.field === 'destination.ip') {
          const ips = (0, _lodash.castArray)(datum.value);
          ips.forEach(ip => {
            const ipType = getIPType(ip);
            processObservable(observablesMap, ip, ipType === 'IPV4' ? _observables.OBSERVABLE_TYPE_IPV4.key : _observables.OBSERVABLE_TYPE_IPV6.key, description);
          });
        }
        // Host name
        if (datum.field === 'host.name') {
          const hostnames = (0, _lodash.castArray)(datum.value);
          hostnames.forEach(name => {
            if (name) {
              processObservable(observablesMap, name, _observables.OBSERVABLE_TYPE_HOSTNAME.key, description);
            }
          });
        }

        // File hash
        if (hashFields.includes(datum.field)) {
          const hashValues = (0, _lodash.castArray)(datum.value);
          hashValues.forEach(hash => {
            processObservable(observablesMap, hash, _observables.OBSERVABLE_TYPE_FILE_HASH.key, description);
          });
        }
        // File path
        if (datum.field === 'file.path') {
          const paths = (0, _lodash.castArray)(datum.value);
          paths.forEach(path => {
            processObservable(observablesMap, path, _observables.OBSERVABLE_TYPE_FILE_PATH.key, description);
          });
        }
        // Domain
        if (datum.field === 'dns.question.name') {
          const names = (0, _lodash.castArray)(datum.value);
          names.forEach(name => {
            processObservable(observablesMap, name, _observables.OBSERVABLE_TYPE_DOMAIN.key, description);
          });
        }
        // Agent ID
        if (datum.field === 'agent.id') {
          const agentIds = (0, _lodash.castArray)(datum.value);
          agentIds.forEach(agentId => {
            processObservable(observablesMap, agentId, _observables.OBSERVABLE_TYPE_AGENT_ID.key, description);
          });
        }
      }
    }
  }

  // remove duplicates of key type and value pairs
  return Array.from(observablesMap.values());
};
exports.getObservablesFromEcs = getObservablesFromEcs;