"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTrendsQuery = exports.getTrends = exports.formatTrends = void 0;
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
var _helpers = require("../../../common/utils/helpers");
var _constants = require("../../../common/constants");
var _mapping_field_util = require("../../lib/mapping_field_util");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getTrendsQuery = policyTemplate => ({
  index: _constants.BENCHMARK_SCORE_INDEX_DEFAULT_NS,
  size: 0,
  sort: '@timestamp:desc',
  query: {
    bool: {
      filter: [{
        term: {
          policy_template: policyTemplate
        }
      }],
      must: [{
        range: {
          '@timestamp': {
            gte: 'now-1d',
            lte: 'now'
          }
        }
      }, {
        term: {
          is_enabled_rules_score: true
        }
      }]
    }
  },
  aggs: {
    by_namespace: {
      terms: {
        field: 'namespace'
      },
      aggs: {
        all_scores: {
          top_hits: {
            size: 100,
            // size: 100, // Maximum top hits result window is 100 which represents > 8 hours of scores samples (CSPM_FINDINGS_STATS_INTERVAL)
            sort: [{
              '@timestamp': {
                order: 'desc'
              }
            }],
            _source: {
              includes: ['@timestamp', 'total_findings', 'passed_findings', 'failed_findings', 'score_by_cluster_id', 'score_by_benchmark_id']
            }
          }
        }
      }
    }
  }
});
exports.getTrendsQuery = getTrendsQuery;
const formatTrends = scoreTrendDocs => {
  return scoreTrendDocs.map(data => {
    return {
      timestamp: data['@timestamp'],
      summary: {
        totalFindings: data.total_findings,
        totalFailed: data.failed_findings,
        totalPassed: data.passed_findings,
        postureScore: (0, _helpers.calculatePostureScore)(data.passed_findings, data.failed_findings)
      },
      clusters: Object.fromEntries(Object.entries(data.score_by_cluster_id).map(([clusterId, cluster]) => [clusterId, {
        totalFindings: cluster.total_findings,
        totalFailed: cluster.failed_findings,
        totalPassed: cluster.passed_findings,
        postureScore: (0, _helpers.calculatePostureScore)(cluster.passed_findings, cluster.failed_findings)
      }])),
      benchmarks: data.score_by_benchmark_id ? Object.fromEntries(Object.entries(data.score_by_benchmark_id).flatMap(([benchmarkId, benchmark]) => Object.entries(benchmark).map(([benchmarkVersion, benchmarkStats]) => {
        const benchmarkIdVersion = (0, _mapping_field_util.toBenchmarkDocFieldKey)(benchmarkId, benchmarkVersion);
        return [benchmarkIdVersion, {
          totalFindings: benchmarkStats.total_findings,
          totalFailed: benchmarkStats.failed_findings,
          totalPassed: benchmarkStats.passed_findings,
          postureScore: (0, _helpers.calculatePostureScore)(benchmarkStats.passed_findings, benchmarkStats.failed_findings)
        }];
      }))) : {}
    };
  });
};
exports.formatTrends = formatTrends;
const getTrends = async (esClient, policyTemplate, logger, namespace = _coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING) => {
  try {
    var _trendsQueryResult$ag, _trendsQueryResult$ag2, _trendsQueryResult$ag3;
    const trendsQueryResult = await esClient.search(getTrendsQuery(policyTemplate));
    if (!((_trendsQueryResult$ag = trendsQueryResult.aggregations) !== null && _trendsQueryResult$ag !== void 0 && (_trendsQueryResult$ag2 = _trendsQueryResult$ag.by_namespace) !== null && _trendsQueryResult$ag2 !== void 0 && _trendsQueryResult$ag2.buckets)) throw new Error('missing trend results from score index');
    const scoreTrendDocs = (_trendsQueryResult$ag3 = trendsQueryResult.aggregations.by_namespace.buckets.map(bucket => {
      var _bucket$all_scores, _bucket$all_scores$hi, _bucket$all_scores$hi2;
      const namespaceKey = bucket.key;
      const documents = ((_bucket$all_scores = bucket.all_scores) === null || _bucket$all_scores === void 0 ? void 0 : (_bucket$all_scores$hi = _bucket$all_scores.hits) === null || _bucket$all_scores$hi === void 0 ? void 0 : (_bucket$all_scores$hi2 = _bucket$all_scores$hi.hits) === null || _bucket$all_scores$hi2 === void 0 ? void 0 : _bucket$all_scores$hi2.map(hit => hit._source)) || [];
      return {
        [namespaceKey]: {
          documents
        }
      };
    })) !== null && _trendsQueryResult$ag3 !== void 0 ? _trendsQueryResult$ag3 : [];
    if (!scoreTrendDocs.length) return {
      trends: [],
      namespaces: []
    }; // No trends data available

    const namespacedData = Object.fromEntries(scoreTrendDocs.map(entry => {
      const [key, value] = Object.entries(entry)[0];
      return [key, value.documents];
    }));
    const namespaceKeys = Object.keys(namespacedData);
    if (!namespacedData[namespace]) {
      logger.warn(`Namespace '${namespace}' not found in trend results.`);
      return {
        trends: [],
        namespaces: namespaceKeys
      };
    }
    return {
      trends: formatTrends(namespacedData[namespace]),
      namespaces: namespaceKeys
    };
  } catch (err) {
    logger.error(`Failed to fetch trendlines data ${err.message}`);
    logger.error(err);
    throw err;
  }
};
exports.getTrends = getTrends;