"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.retentionPolicyConfigSchema = exports.GenericRetentionPolicy = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _path = require("path");
var _configSchema = require("@kbn/config-schema");
var _fs = require("./fs");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const retentionPolicyConfigSchema = exports.retentionPolicyConfigSchema = _configSchema.schema.object({
  maxFiles: _configSchema.schema.maybe(_configSchema.schema.number({
    min: 1,
    max: 365
  })),
  maxAccumulatedFileSize: _configSchema.schema.maybe(_configSchema.schema.byteSize()),
  removeOlderThan: _configSchema.schema.maybe(_configSchema.schema.duration({
    max: '365d'
  }))
}, {
  validate: config => {
    if (!config.maxFiles && !config.maxAccumulatedFileSize && !config.removeOlderThan) {
      return 'Retention policy must define at least one directive: maxFiles, maxAccumulatedFileSize or removeOlderThan';
    }
  }
});
class GenericRetentionPolicy {
  constructor(config, context) {
    (0, _defineProperty2.default)(this, "logFileFolder", void 0);
    this.config = config;
    this.context = context;
    this.logFileFolder = (0, _path.dirname)(this.context.filePath);
  }
  async apply() {
    const {
      maxFiles,
      maxAccumulatedFileSize,
      removeOlderThan
    } = this.config;
    const orderedFilesBaseNames = await this.context.getOrderedRolledFiles();
    const absOrderedFiles = orderedFilesBaseNames.map(filepath => (0, _path.join)(this.logFileFolder, filepath));
    const filesToDelete = new Set();
    if (maxFiles) {
      const exceedingFiles = absOrderedFiles.slice(maxFiles, absOrderedFiles.length);
      exceedingFiles.forEach(file => filesToDelete.add(file));
    }
    if (maxAccumulatedFileSize) {
      const toRemoveByFileSize = await (0, _utils.listFilesExceedingSize)({
        orderedFiles: absOrderedFiles,
        maxSizeInBytes: maxAccumulatedFileSize.getValueInBytes()
      });
      toRemoveByFileSize.forEach(file => filesToDelete.add(file));
    }
    if (removeOlderThan) {
      const toRemoveByAge = await (0, _utils.listFilesOlderThan)({
        orderedFiles: absOrderedFiles,
        duration: removeOlderThan
      });
      toRemoveByAge.forEach(file => filesToDelete.add(file));
    }
    if (filesToDelete.size > 0) {
      await (0, _fs.deleteFiles)({
        filesToDelete: [...filesToDelete]
      });
    }
  }
}
exports.GenericRetentionPolicy = GenericRetentionPolicy;