"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PrepareIndicesForAssistantGraphEvaluations = void 0;
var _prepare_indices_for_evalutations = require("../../prepare_indices_for_evalutations");
var _indices_create_requests = require("./indices_create_requests");
var _index_requests = require("./index_requests");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ENVIRONMENTS = ['production', 'staging', 'development'];
class PrepareIndicesForAssistantGraphEvaluations extends _prepare_indices_for_evalutations.PrepareIndicesForEvaluations {
  constructor({
    esClient,
    logger
  }) {
    super({
      esClient,
      indicesCreateRequests: PrepareIndicesForAssistantGraphEvaluations.hydrateRequestTemplate(Object.values(_indices_create_requests.indicesCreateRequests)),
      indexRequests: PrepareIndicesForAssistantGraphEvaluations.hydrateRequestTemplate(Object.values(_index_requests.indexRequests)),
      logger
    });
  }
  static hydrateRequestTemplate(requests) {
    return requests.map(request => {
      return ENVIRONMENTS.map(environment => {
        return {
          ...request,
          index: request.index.replace(/\[environment\]/g, environment).replace(/\[date\]/g, this.getRandomDate())
        };
      });
    }).flat();
  }
  async cleanup() {
    this.logger.debug('Deleting assistant indices for evaluations');
    const requests = [...Object.values(_indices_create_requests.indicesCreateRequests), ...Object.values(_index_requests.indexRequests)];
    const indexPatternsToDelete = Object.values(requests).map(index => index.index.replace(/\[environment\]/g, '*').replace(/\[date\]/g, '*'));
    const indicesResolveIndexResponses = await Promise.all(indexPatternsToDelete.map(async indexPattern => this.esClient.indices.resolveIndex({
      name: indexPattern,
      expand_wildcards: 'open'
    })));
    const indicesToDelete = indicesResolveIndexResponses.flatMap(response => response.indices).map(index => index.name);
    const dataStreamsToDelete = indicesResolveIndexResponses.flatMap(response => response.data_streams).map(dataStream => dataStream.name);
    if (indicesToDelete.length > 0) {
      this.logger.info('Deleting indices');
      await this.esClient.indices.delete({
        index: indicesToDelete
      });
    }
    if (dataStreamsToDelete.length > 0) {
      this.logger.info('Deleting data streams');
      await this.esClient.indices.deleteDataStream({
        name: dataStreamsToDelete
      });
    }
  }
  static getRandomDate() {
    const year = Math.floor(Math.random() * (2050 - 2000 + 1)) + 2000;
    const month = String(Math.floor(Math.random() * 12) + 1).padStart(2, '0');
    const day = String(Math.floor(Math.random() * 28) + 1).padStart(2, '0');
    return `${year}.${month}.${day}`;
  }
}
exports.PrepareIndicesForAssistantGraphEvaluations = PrepareIndicesForAssistantGraphEvaluations;