"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parse = exports.Parser = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _antlr = require("antlr4");
var _esql_error_listener = require("./esql_error_listener");
var _formatting = require("./formatting");
var _builder = require("../builder");
var _cst_to_ast_converter = require("./cst_to_ast_converter");
var _esql_lexer = _interopRequireDefault(require("../antlr/esql_lexer"));
var _esql_parser = _interopRequireDefault(require("../antlr/esql_parser"));
var _is = require("../ast/is");
var _utils = require("../visitor/utils");
var _constants = require("./constants");
var _Parser;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
class Parser {
  constructor(src, options = {}) {
    (0, _defineProperty2.default)(this, "streams", void 0);
    (0, _defineProperty2.default)(this, "lexer", void 0);
    (0, _defineProperty2.default)(this, "tokens", void 0);
    (0, _defineProperty2.default)(this, "parser", void 0);
    (0, _defineProperty2.default)(this, "errors", new _esql_error_listener.ESQLErrorListener());
    this.src = src;
    this.options = options;
    const streams = this.streams = _antlr.CharStreams.fromString(src);
    const lexer = this.lexer = new _esql_lexer.default(streams);
    const tokens = this.tokens = new _antlr.CommonTokenStream(lexer);
    const parser = this.parser = new _esql_parser.default(tokens);
    lexer.removeErrorListeners();
    lexer.addErrorListener(this.errors);
    parser.removeErrorListeners();
    parser.addErrorListener(this.errors);
  }
  parseTarget([rule, conversion]) {
    const ctx = this.parser[rule].call(this.parser);
    const converter = new _cst_to_ast_converter.CstToAstConverter(this);
    const root = converter[conversion].call(converter, ctx);
    if (!root) {
      throw new Error('Parsing failed: no root node found');
    }
    const errors = this.errors.getErrors();
    if (this.options.withFormatting && (0, _is.isQuery)(root)) {
      const decorations = (0, _formatting.collectDecorations)(this.tokens);
      (0, _formatting.attachDecorations)(root, this.tokens.tokens, decorations.lines);
    }
    const result = {
      root,
      errors,
      tokens: this.tokens.tokens,
      // @deprecated Use `root` instead.
      ast: root.commands
    };
    return result;
  }
  parseSourceCommand() {
    return this.parseTarget(['sourceCommand', 'fromSourceCommand']);
  }
  parseProcessingCommand() {
    return this.parseTarget(['processingCommand', 'fromProcessingCommand']);
  }
  parse() {
    try {
      return this.parseTarget(['singleStatement', 'fromSingleStatement']);
    } catch (error) {
      if (error !== 'Empty Stack')
        // eslint-disable-next-line no-console
        console.error(error);
      const root = _builder.Builder.expression.query();
      return {
        root,
        ast: root.commands,
        errors: [{
          startLineNumber: 0,
          endLineNumber: 0,
          startColumn: 0,
          endColumn: 0,
          message: `Invalid query [${this.src}]`,
          severity: 'error',
          code: 'parseError'
        }],
        tokens: []
      };
    }
  }
  parseErrors() {
    this.parser.singleStatement();
    return this.errors.getErrors();
  }
}

/**
 * @deprecated Use `Parser.parse` instead.
 */
exports.Parser = Parser;
_Parser = Parser;
(0, _defineProperty2.default)(Parser, "create", (src, options) => {
  return new _Parser(src, options);
});
/**
 * Parse a complete ES|QL query, generating an AST and a list of parsing errors.
 *
 * Make sure to check the returned `errors` list for any parsing issues.
 *
 * For example:
 *
 * ```typescript
 * const result = Parser.parse('FROM my_index | STATS count(*)');
 * ```
 *
 * @param src Source text to parse.
 * @param options Parsing options.
 */
(0, _defineProperty2.default)(Parser, "parse", (src, options) => {
  return _Parser.create(src, options).parse();
});
/**
 * Extract parsing errors from the source text without generating an AST.
 *
 * @param src Source text to parse for errors.
 * @returns A list of parsing errors.
 */
(0, _defineProperty2.default)(Parser, "parseErrors", src => {
  return _Parser.create(src).parseErrors();
});
(0, _defineProperty2.default)(Parser, "parseQuery", (src, options) => {
  const result = _Parser.parse(src, options);
  if (result.errors.length) {
    throw result.errors[0];
  }
  return result;
});
/**
 * Parse a single ES|QL command, generating an AST and a list of parsing errors.
 *
 * Make sure to check the returned `errors` list for any parsing issues.
 *
 * For example:
 *
 * ```typescript
 * const result = Parser.parseCommand('ROW abc = 123');
 * ```
 *
 * @param src Source text of a single command to parse.
 * @param options Parsing options.
 * @returns A result object containing the parsed command, its AST, tokens, and errors.
 */
(0, _defineProperty2.default)(Parser, "parseCommand", (src, options) => {
  const [token] = _Parser.tokens(src, 1);
  if (!token || token.type === _esql_lexer.default.EOF) {
    throw new Error('Cannot parse empty command');
  }
  const isSourceCommand = _constants.SOURCE_COMMANDS.has(token.text.toUpperCase());
  const result = _Parser.parse(isSourceCommand ? src : 'FROM a|' + src, options);
  if (!result.errors.length) {
    const commands = result.root.commands;
    const end = isSourceCommand ? 0 : 1;
    if (end + 1 !== commands.length) {
      throw new Error(`Could not parse a single command completely: "${src}". `);
    }
    const command = commands[end];
    if ((0, _is.isCommand)(command)) {
      return {
        ...result,
        root: command,
        ast: [command]
      };
    }
  }
  throw new Error(`Invalid command: ${src}`);
});
/**
 * Parse a single ES|QL expression, generating an AST and a list of parsing errors.
 *
 * Make sure to check the returned `errors` list for any parsing issues.
 *
 * For example:
 *
 * ```typescript
 * const result = Parser.parseExpression('count(*) + 1');
 * ```
 *
 * @param src Source text of an expression to parse.
 * @param options Parsing options.
 * @returns A result object containing the parsed expression, its AST, tokens, and errors.
 */
(0, _defineProperty2.default)(Parser, "parseExpression", (src, options) => {
  const [token] = _Parser.tokens(src, 1);
  if (!token || token.type === _esql_lexer.default.EOF) {
    throw new Error('Cannot parse empty command');
  }
  if (token.text[0] === '{') {
    return _Parser.parseMap(src, options);
  }
  const {
    root,
    ast,
    errors,
    ...result
  } = _Parser.parseCommand('EVAL ' + src, options);
  const expressions = [...(0, _utils.singleItems)(root.args)];
  if (expressions.length !== 1) {
    throw new Error(`Invalid expression: expected a single expression, got ${expressions.length} in "${src}"`);
  }
  const expression = expressions[0];
  if (!(0, _is.isProperNode)(expression)) {
    throw errors[0] || new Error('Invalid expression: ' + src);
  }
  return {
    root: expression,
    errors,
    ...result,
    // @deprecated Use `root` instead.
    ast: undefined
  };
});
(0, _defineProperty2.default)(Parser, "parseMap", (src, options) => {
  const {
    root,
    ast,
    errors,
    ...result
  } = _Parser.parseCommand('ROW f(1,' + src + ')', options);
  const expressions = [...(0, _utils.singleItems)(root.args)];
  if (expressions.length !== 1) {
    throw new Error(`Invalid expression: expected a single expression, got ${expressions.length} in "${src}"`);
  }
  const fn = expressions[0];
  if (!(0, _is.isFunctionExpression)(fn)) {
    throw errors[0] || new Error('Invalid expression: ' + src);
  }
  const map = fn.args[1];
  if (!(0, _is.isMap)(map)) {
    throw errors[0] || new Error('Invalid expression: ' + src);
  }
  return {
    root: map,
    errors,
    ...result,
    // @deprecated Use `root` instead.
    ast: undefined
  };
});
/**
 * Get the first `count` tokens from the source text.
 *
 * @param src Text to parse for tokens.
 * @param count Number of tokens to parse.
 * @param visible Whether to return only visible tokens (not comments or whitespace).
 * @returns An array of parsed tokens.
 */
(0, _defineProperty2.default)(Parser, "tokens", (src, count, visible = true) => {
  const streams = _antlr.CharStreams.fromString(src);
  const lexer = new _esql_lexer.default(streams);
  const tokens = [];
  let i = 0;
  while (i < count) {
    const token = lexer.nextToken();
    if (token.type === _esql_lexer.default.EOF) {
      break;
    }
    if (visible && token.channel !== _constants.DEFAULT_CHANNEL) {
      continue;
    }
    tokens.push(token);
    i++;
  }
  return tokens;
});
const parse = (src, options = {}) => {
  if (src == null) {
    const commands = [];
    return {
      ast: commands,
      root: _builder.Builder.expression.query(commands),
      errors: [],
      tokens: []
    };
  }
  return Parser.create(src, options).parse();
};
exports.parse = parse;