"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FleetActionsClient = void 0;
var _errors = require("../../../common/errors");
var _actions = require("./actions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class FleetActionsClient {
  constructor(esClient, packageName) {
    this.esClient = esClient;
    this.packageName = packageName;
    if (!packageName) {
      throw new _errors.FleetActionsClientError('packageName is required');
    }
  }
  _verifyAction(action) {
    if (action.input_type !== this.packageName) {
      throw new _errors.FleetActionsClientError(`Action package name mismatch. Expected "${this.packageName}" got "${action.input_type}. Action: ${action.action_id}."`);
    }
    return action;
  }
  _verifyResult(result) {
    if (result.action_input_type !== this.packageName) {
      throw new _errors.FleetActionsClientError(`Action result package name mismatch. Expected "${this.packageName}" got "${result.action_input_type}". Result: ${result.action_id}`);
    }
    return result;
  }
  async create(action) {
    const verifiedAction = this._verifyAction(action);
    return (0, _actions.createAction)(this.esClient, verifiedAction);
  }
  async bulkCreate(actions) {
    actions.map(action => this._verifyAction(action));
    return (0, _actions.bulkCreateActions)(this.esClient, actions);
  }
  async getActionsByIds(ids) {
    const actions = await (0, _actions.getActionsByIds)(this.esClient, ids);
    actions.items.every(action => this._verifyAction(action));
    return actions;
  }
  async getActionsWithKuery(kuery) {
    const actions = await (0, _actions.getActionsWithKuery)(this.esClient, kuery);
    actions.items.every(action => this._verifyAction(action));
    return actions;
  }
  async getResultsByIds(ids) {
    const results = await (0, _actions.getActionResultsByIds)(this.esClient, ids);
    results.items.every(result => this._verifyResult(result));
    return results;
  }
  async getResultsWithKuery(kuery) {
    const results = await (0, _actions.getActionResultsWithKuery)(this.esClient, kuery);
    results.items.every(result => this._verifyResult(result));
    return results;
  }
}
exports.FleetActionsClient = FleetActionsClient;