"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchAndAssignAgentMetrics = fetchAndAssignAgentMetrics;
var _app_context = require("../app_context");
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const AGGREGATION_MAX_SIZE = 1000;
async function fetchAndAssignAgentMetrics(esClient, agents) {
  try {
    return await _fetchAndAssignAgentMetrics(esClient, agents);
  } catch (err) {
    //  Do not throw if we are not able to fetch metrics, as it could occur if the user is missing some permissions
    _app_context.appContextService.getLogger().warn(err);
    return agents;
  }
}
async function _fetchAndAssignAgentMetrics(esClient, agents) {
  var _res$aggregations$age, _res$aggregations;
  const res = await esClient.search({
    ...aggregationQueryBuilder(agents.map(({
      id
    }) => id)),
    index: 'metrics-elastic_agent.*'
  });
  const formattedResults = (_res$aggregations$age = (_res$aggregations = res.aggregations) === null || _res$aggregations === void 0 ? void 0 : _res$aggregations.agents.buckets.reduce((acc, bucket) => {
    acc[bucket.key] = {
      sum_memory_size: bucket.sum_memory_size.value,
      sum_cpu: bucket.sum_cpu.value
    };
    return acc;
  }, {})) !== null && _res$aggregations$age !== void 0 ? _res$aggregations$age : {};
  return agents.map(agent => {
    const results = formattedResults[agent.id];
    return {
      ...agent,
      metrics: {
        cpu_avg: results !== null && results !== void 0 && results.sum_cpu ? Math.trunc(results.sum_cpu * 100000) / 100000 : undefined,
        memory_size_byte_avg: results !== null && results !== void 0 && results.sum_memory_size ? Math.trunc(results === null || results === void 0 ? void 0 : results.sum_memory_size) : undefined
      }
    };
  });
}
const aggregationQueryBuilder = agentIds => ({
  size: 0,
  query: {
    bool: {
      must: [{
        terms: {
          _tier: _constants.DATA_TIERS
        }
      }, {
        range: {
          '@timestamp': {
            gte: 'now-5m'
          }
        }
      }, {
        terms: {
          'elastic_agent.id': agentIds
        }
      }, {
        bool: {
          filter: [{
            bool: {
              should: [{
                term: {
                  'data_stream.dataset': 'elastic_agent.elastic_agent'
                }
              }]
            }
          }]
        }
      }]
    }
  },
  aggs: {
    agents: {
      terms: {
        field: 'elastic_agent.id',
        size: AGGREGATION_MAX_SIZE
      },
      aggs: {
        sum_memory_size: {
          sum_bucket: {
            buckets_path: 'processes>avg_memory_size'
          }
        },
        sum_cpu: {
          sum_bucket: {
            buckets_path: 'processes>avg_cpu'
          }
        },
        processes: {
          terms: {
            field: 'component.id',
            size: AGGREGATION_MAX_SIZE,
            order: {
              _count: 'desc'
            }
          },
          aggs: {
            avg_cpu: {
              avg_bucket: {
                buckets_path: 'cpu_time_series>cpu'
              }
            },
            avg_memory_size: {
              avg: {
                field: 'system.process.memory.size'
              }
            },
            cpu_time_series: {
              date_histogram: {
                field: '@timestamp',
                calendar_interval: 'minute'
              },
              aggs: {
                max_cpu: {
                  max: {
                    field: 'system.process.cpu.total.value'
                  }
                },
                cpu_derivative: {
                  derivative: {
                    buckets_path: 'max_cpu',
                    gap_policy: 'skip',
                    unit: '10s'
                  }
                },
                cpu: {
                  bucket_script: {
                    buckets_path: {
                      cpu_total: 'cpu_derivative[normalized_value]'
                    },
                    script: {
                      source: `if (params.cpu_total > 0) {
                      return params.cpu_total / params._interval;
                    } else {
                      return 0;
                    }`,
                      lang: 'painless',
                      params: {
                        _interval: 10000
                      }
                    },
                    gap_policy: 'skip'
                  }
                }
              }
            }
          }
        }
      }
    }
  }
});