"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.cloudConnectorService = exports.CloudConnectorService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _constants = require("../../common/constants");
var _errors = require("../errors");
var _app_context = require("./app_context");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; } /*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
class CloudConnectorService {
  getLogger(...childContextPaths) {
    return _app_context.appContextService.getLogger().get('CloudConnectorService', ...childContextPaths);
  }
  async create(soClient, cloudConnector) {
    const logger = this.getLogger('create');
    try {
      var _vars$role_arn, _vars$role_arn2, _vars$external_id;
      logger.info('Creating cloud connector');
      this.validateCloudConnectorDetails(cloudConnector);
      const {
        vars,
        cloudProvider
      } = cloudConnector;
      if (!vars || Object.keys(vars).length === 0) {
        logger.error(`Package policy must contain ${cloudProvider} input vars`);
        throw new _errors.CloudConnectorCreateError(`CloudConnectorService Package policy must contain ${cloudProvider} input vars`);
      }
      const name = cloudConnector.cloudProvider === 'aws' && (_vars$role_arn = vars.role_arn) !== null && _vars$role_arn !== void 0 && _vars$role_arn.value ? vars.role_arn.value : cloudConnector.name;

      // Check if space awareness is enabled for namespace handling
      const {
        isSpaceAwarenessEnabled
      } = await Promise.resolve().then(() => _interopRequireWildcard(require('./spaces/helpers')));
      const useSpaceAwareness = await isSpaceAwarenessEnabled();
      const namespace = useSpaceAwareness ? '*' : undefined;

      // Create cloud connector saved object
      const cloudConnectorAttributes = {
        name,
        namespace,
        cloudProvider,
        vars: {
          ...(((_vars$role_arn2 = vars.role_arn) === null || _vars$role_arn2 === void 0 ? void 0 : _vars$role_arn2.value) && {
            role_arn: vars.role_arn
          }),
          ...(((_vars$external_id = vars.external_id) === null || _vars$external_id === void 0 ? void 0 : _vars$external_id.value) && {
            external_id: vars.external_id
          })
        },
        packagePolicyCount: 1,
        created_at: new Date().toISOString(),
        updated_at: new Date().toISOString()
      };
      const savedObject = await soClient.create(_constants.CLOUD_CONNECTOR_SAVED_OBJECT_TYPE, cloudConnectorAttributes);
      logger.info('Successfully created cloud connector');
      return {
        id: savedObject.id,
        ...savedObject.attributes
      };
    } catch (error) {
      logger.error('Failed to create cloud connector', error.message);
      throw new _errors.CloudConnectorCreateError(`CloudConnectorService Failed to create cloud connector: ${error.message}\n${error.stack}`);
    }
  }
  async getList(soClient, options) {
    const logger = this.getLogger('getList');
    logger.debug('Getting cloud connectors list');
    try {
      const cloudConnectors = await soClient.find({
        type: _constants.CLOUD_CONNECTOR_SAVED_OBJECT_TYPE,
        page: (options === null || options === void 0 ? void 0 : options.page) || 1,
        perPage: (options === null || options === void 0 ? void 0 : options.perPage) || 20,
        sortField: 'created_at',
        sortOrder: 'desc'
      });
      logger.debug('Successfully retrieved cloud connectors list');
      return cloudConnectors.saved_objects.map(savedObject => ({
        id: savedObject.id,
        ...savedObject.attributes
      }));
    } catch (error) {
      logger.error('Failed to get cloud connectors list', error.message);
      throw new _errors.CloudConnectorGetListError(`Failed to get cloud connectors list: ${error.message}\n${error.stack}`);
    }
  }
  async getById(soClient, cloudConnectorId) {
    const logger = this.getLogger('getById');
    try {
      logger.info(`Getting cloud connector ${cloudConnectorId}`);
      const cloudConnector = await soClient.get(_constants.CLOUD_CONNECTOR_SAVED_OBJECT_TYPE, cloudConnectorId);
      logger.info(`Successfully retrieved cloud connector ${cloudConnectorId}`);
      return {
        id: cloudConnector.id,
        ...cloudConnector.attributes
      };
    } catch (error) {
      logger.error('Failed to get cloud connector', error.message);
      throw new _errors.CloudConnectorGetListError(`Failed to get cloud connector: ${error.message}\n${error.stack}`);
    }
  }
  async update(soClient, cloudConnectorId, updates) {
    const logger = this.getLogger('update');
    try {
      logger.info(`Updating cloud connector ${cloudConnectorId}`);

      // Get existing cloud connector
      const existingCloudConnector = await soClient.get(_constants.CLOUD_CONNECTOR_SAVED_OBJECT_TYPE, cloudConnectorId);

      // Validate updates if vars are provided
      if (updates.vars) {
        const tempCloudConnector = {
          name: updates.name || existingCloudConnector.attributes.name,
          vars: updates.vars,
          cloudProvider: existingCloudConnector.attributes.cloudProvider
        };
        this.validateCloudConnectorDetails(tempCloudConnector);
      }

      // Prepare update attributes
      const updateAttributes = {
        updated_at: new Date().toISOString()
      };
      if (updates.name) {
        updateAttributes.name = updates.name;
      }
      if (updates.vars) {
        var _updates$vars$role_ar, _updates$vars$externa;
        updateAttributes.vars = {
          ...(((_updates$vars$role_ar = updates.vars.role_arn) === null || _updates$vars$role_ar === void 0 ? void 0 : _updates$vars$role_ar.value) && {
            role_arn: updates.vars.role_arn
          }),
          ...(((_updates$vars$externa = updates.vars.external_id) === null || _updates$vars$externa === void 0 ? void 0 : _updates$vars$externa.value) && {
            external_id: updates.vars.external_id
          })
        };
      }

      // Update the saved object
      const updatedSavedObject = await soClient.update(_constants.CLOUD_CONNECTOR_SAVED_OBJECT_TYPE, cloudConnectorId, updateAttributes);
      logger.info(`Successfully updated cloud connector ${cloudConnectorId}`);

      // Return the updated cloud connector with merged attributes
      const mergedAttributes = {
        ...existingCloudConnector.attributes,
        ...updatedSavedObject.attributes
      };
      return {
        id: cloudConnectorId,
        ...mergedAttributes
      };
    } catch (error) {
      logger.error('Failed to update cloud connector', error.message);
      throw new _errors.CloudConnectorCreateError(`Failed to update cloud connector: ${error.message}\n${error.stack}`);
    }
  }
  async delete(soClient, cloudConnectorId, force = false) {
    const logger = this.getLogger('delete');
    try {
      logger.info(`Deleting cloud connector ${cloudConnectorId} (force: ${force})`);

      // First, get the cloud connector to check packagePolicyCount
      const cloudConnector = await soClient.get(_constants.CLOUD_CONNECTOR_SAVED_OBJECT_TYPE, cloudConnectorId);

      // Check if cloud connector is still in use by package policies (unless force is true)
      if (!force && cloudConnector.attributes.packagePolicyCount > 0) {
        const errorMessage = `Cannot delete cloud connector "${cloudConnector.attributes.name}" as it is being used by ${cloudConnector.attributes.packagePolicyCount} package policies`;
        logger.error(errorMessage);
        throw new _errors.CloudConnectorDeleteError(errorMessage);
      }

      // Log a warning if force deleting a connector that's still in use
      if (force && cloudConnector.attributes.packagePolicyCount > 0) {
        logger.warn(`Force deleting cloud connector "${cloudConnector.attributes.name}" which is still being used by ${cloudConnector.attributes.packagePolicyCount} package policies`);
      }

      // Delete the cloud connector
      await soClient.delete(_constants.CLOUD_CONNECTOR_SAVED_OBJECT_TYPE, cloudConnectorId);
      logger.info(`Successfully deleted cloud connector ${cloudConnectorId}`);
      return {
        id: cloudConnectorId
      };
    } catch (error) {
      logger.error('Failed to delete cloud connector', error.message);

      // Re-throw CloudConnectorDeleteError as-is to preserve the original error message
      if (error instanceof _errors.CloudConnectorDeleteError) {
        throw error;
      }
      throw new _errors.CloudConnectorDeleteError(`Failed to delete cloud connector: ${error.message}\n${error.stack}`);
    }
  }
  validateCloudConnectorDetails(cloudConnector) {
    const logger = this.getLogger('validate cloud connector details');
    const vars = cloudConnector.vars;
    if (cloudConnector.cloudProvider === 'aws') {
      var _vars$role_arn3, _vars$external_id2;
      const roleArn = (_vars$role_arn3 = vars.role_arn) === null || _vars$role_arn3 === void 0 ? void 0 : _vars$role_arn3.value;
      if (!roleArn) {
        logger.error('Package policy must contain role_arn variable');
        throw new _errors.CloudConnectorInvalidVarsError('Package policy must contain role_arn variable');
      }
      const externalId = (_vars$external_id2 = vars.external_id) === null || _vars$external_id2 === void 0 ? void 0 : _vars$external_id2.value;
      if (!externalId) {
        logger.error('Package policy must contain valid external_id secret reference');
        throw new _errors.CloudConnectorInvalidVarsError('Package policy must contain valid external_id secret reference');
      }
      const isValidExternalId = (externalId === null || externalId === void 0 ? void 0 : externalId.id) && (externalId === null || externalId === void 0 ? void 0 : externalId.isSecretRef) && CloudConnectorService.EXTERNAL_ID_REGEX.test(externalId.id);
      if (!isValidExternalId) {
        logger.error('External ID secret reference must be a valid secret reference');
        throw new _errors.CloudConnectorInvalidVarsError('External ID secret reference is not valid');
      }
    } else {
      logger.error(`Unsupported cloud provider: ${cloudConnector.cloudProvider}`);
      throw new _errors.CloudConnectorCreateError(`Unsupported cloud provider: ${cloudConnector.cloudProvider}`);
    }
  }
}
exports.CloudConnectorService = CloudConnectorService;
(0, _defineProperty2.default)(CloudConnectorService, "EXTERNAL_ID_REGEX", /^[a-zA-Z0-9_-]{20}$/);
const cloudConnectorService = exports.cloudConnectorService = new CloudConnectorService();