"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.cleanUpKibanaAssetsStep = cleanUpKibanaAssetsStep;
exports.cleanUpUnusedKibanaAssetsStep = cleanUpUnusedKibanaAssetsStep;
exports.stepInstallKibanaAssets = stepInstallKibanaAssets;
exports.stepInstallKibanaAssetsWithStreaming = stepInstallKibanaAssetsWithStreaming;
var _install = require("../../../kibana/assets/install");
var _utils = require("../../utils");
var _remove = require("../../remove");
var _types = require("../../../../../../common/types");
var _install_with_streaming = require("../../../kibana/assets/install_with_streaming");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function stepInstallKibanaAssets(context) {
  const {
    savedObjectsClient,
    logger,
    installedPkg,
    packageInstallContext,
    spaceId
  } = context;
  const {
    packageInfo
  } = packageInstallContext;
  const {
    name: pkgName,
    title: pkgTitle
  } = packageInfo;
  const kibanaAssetPromise = (0, _utils.withPackageSpan)('Install Kibana assets', () => (0, _install.installKibanaAssetsAndReferencesMultispace)({
    savedObjectsClient,
    pkgName,
    pkgTitle,
    packageInstallContext,
    installedPkg,
    logger,
    spaceId,
    assetTags: packageInfo === null || packageInfo === void 0 ? void 0 : packageInfo.asset_tags
  }));
  // Necessary to avoid async promise rejection warning
  // See https://stackoverflow.com/questions/40920179/should-i-refrain-from-handling-promise-rejection-asynchronously
  kibanaAssetPromise.catch(() => {});
  return {
    kibanaAssetPromise
  };
}
async function stepInstallKibanaAssetsWithStreaming(context) {
  const {
    savedObjectsClient,
    packageInstallContext,
    spaceId
  } = context;
  const {
    packageInfo
  } = packageInstallContext;
  const {
    name: pkgName
  } = packageInfo;
  const installedKibanaAssetsRefs = await (0, _utils.withPackageSpan)('Install Kibana assets with streaming', () => (0, _install_with_streaming.installKibanaAssetsWithStreaming)({
    savedObjectsClient,
    pkgName,
    packageInstallContext,
    spaceId
  }));
  return {
    installedKibanaAssetsRefs
  };
}
async function cleanUpKibanaAssetsStep(context) {
  var _installedPkg$attribu;
  const {
    logger,
    installedPkg,
    packageInstallContext,
    spaceId,
    retryFromLastState,
    force,
    initialState
  } = context;
  const {
    packageInfo
  } = packageInstallContext;

  // In case of retry clean up previous installed kibana assets
  if (!force && retryFromLastState && initialState === _types.INSTALL_STATES.INSTALL_KIBANA_ASSETS && installedPkg !== null && installedPkg !== void 0 && (_installedPkg$attribu = installedPkg.attributes) !== null && _installedPkg$attribu !== void 0 && _installedPkg$attribu.installed_kibana && installedPkg.attributes.installed_kibana.length > 0) {
    var _installedObjects;
    let {
      installed_kibana: installedObjects
    } = installedPkg.attributes;
    logger.debug('Retry transition - clean up Kibana assets first');

    // Do not delete creatd alerting rules
    installedObjects = ((_installedObjects = installedObjects) !== null && _installedObjects !== void 0 ? _installedObjects : []).filter(asset => asset.type !== _types.KibanaSavedObjectType.alert);
    await (0, _remove.deleteKibanaAssets)({
      installedObjects,
      spaceId,
      packageSpecConditions: packageInfo === null || packageInfo === void 0 ? void 0 : packageInfo.conditions,
      logger
    });
  }
}

/**
 * Cleans up Kibana assets that are no longer in the package. As opposite to
 * `cleanUpKibanaAssetsStep`, this one is used after the package assets are
 * installed.
 *
 * This function compares the currently installed Kibana assets with the assets
 * in the previous package and removes any assets that are no longer present in the
 * new installation.
 *
 */
async function cleanUpUnusedKibanaAssetsStep(context) {
  var _installedPkg$attribu2;
  const {
    logger,
    installedPkg,
    packageInstallContext,
    spaceId,
    installedKibanaAssetsRefs
  } = context;
  const {
    packageInfo
  } = packageInstallContext;
  if (!installedKibanaAssetsRefs) {
    return;
  }
  logger.debug('Clean up Kibana assets that are no longer in the package');

  // Get the assets installed by the previous package
  const previousAssetRefs = (_installedPkg$attribu2 = installedPkg === null || installedPkg === void 0 ? void 0 : installedPkg.attributes.installed_kibana) !== null && _installedPkg$attribu2 !== void 0 ? _installedPkg$attribu2 : [];

  // Remove any assets that are not in the new package
  const nextAssetRefKeys = new Set(installedKibanaAssetsRefs.map(asset => `${asset.id}-${asset.type}`));
  const assetsToRemove = previousAssetRefs.filter(existingAsset => !nextAssetRefKeys.has(`${existingAsset.id}-${existingAsset.type}`));
  if (assetsToRemove.length === 0) {
    return;
  }
  await (0, _utils.withPackageSpan)('Clean up Kibana assets that are no longer in the package', async () => {
    await (0, _remove.deleteKibanaAssets)({
      installedObjects: assetsToRemove,
      spaceId,
      packageSpecConditions: packageInfo === null || packageInfo === void 0 ? void 0 : packageInfo.conditions,
      logger
    });
  });
}