"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteOutputSecrets = deleteOutputSecrets;
exports.extractAndUpdateOutputSecrets = extractAndUpdateOutputSecrets;
exports.extractAndWriteOutputSecrets = extractAndWriteOutputSecrets;
exports.getOutputSecretReferences = getOutputSecretReferences;
exports.isOutputSecretStorageEnabled = isOutputSecretStorageEnabled;
var _constants = require("../../constants");
var _app_context = require("../app_context");
var _ = require("..");
var _fleet_server = require("../fleet_server");
var _common = require("./common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function isOutputSecretStorageEnabled(esClient, soClient) {
  var _appContextService$ge, _appContextService$ge2, _appContextService$ge3;
  const logger = _app_context.appContextService.getLogger();

  // if serverless then output secrets will always be supported
  const isFleetServerStandalone = (_appContextService$ge = (_appContextService$ge2 = _app_context.appContextService.getConfig()) === null || _appContextService$ge2 === void 0 ? void 0 : (_appContextService$ge3 = _appContextService$ge2.internal) === null || _appContextService$ge3 === void 0 ? void 0 : _appContextService$ge3.fleetServerStandalone) !== null && _appContextService$ge !== void 0 ? _appContextService$ge : false;
  if (isFleetServerStandalone) {
    logger.trace('Output secrets storage is enabled as fleet server is standalone');
    return true;
  }

  // now check the flag in settings to see if the fleet server requirement has already been met
  // once the requirement has been met, output secrets are always on
  const settings = await _.settingsService.getSettingsOrUndefined(soClient);
  if (settings && settings.output_secret_storage_requirements_met) {
    logger.debug('Output secrets storage requirements already met, turned on in settings');
    return true;
  }

  // otherwise check if we have the minimum fleet server version and enable secrets if so
  if (await (0, _fleet_server.checkFleetServerVersionsForSecretsStorage)(esClient, soClient, _constants.OUTPUT_SECRETS_MINIMUM_FLEET_SERVER_VERSION)) {
    logger.debug('Enabling output secrets storage as minimum fleet server version has been met');
    try {
      await _.settingsService.saveSettings(soClient, {
        output_secret_storage_requirements_met: true
      });
    } catch (err) {
      // we can suppress this error as it will be retried on the next function call
      logger.warn(`Failed to save settings after enabling output secrets storage: ${err.message}`);
    }
    return true;
  }
  logger.info('Secrets storage is disabled as minimum fleet server version has not been met');
  return false;
}
async function extractAndWriteOutputSecrets(opts) {
  const {
    output,
    esClient,
    secretHashes = {}
  } = opts;
  const secretPaths = getOutputSecretPaths(output.type, output).filter(path => typeof path.value === 'string');
  const secretRes = await (0, _common.extractAndWriteSOSecrets)({
    soObject: output,
    secretPaths,
    esClient,
    secretHashes
  });
  return {
    output: secretRes.soObjectWithSecrets,
    secretReferences: secretRes.secretReferences
  };
}
async function extractAndUpdateOutputSecrets(opts) {
  const {
    oldOutput,
    outputUpdate,
    esClient,
    secretHashes
  } = opts;
  const outputType = outputUpdate.type || oldOutput.type;
  const oldSecretPaths = getOutputSecretPaths(oldOutput.type, oldOutput);
  const updatedSecretPaths = getOutputSecretPaths(outputType, outputUpdate);
  const secretRes = await (0, _common.extractAndUpdateSOSecrets)({
    updatedSoObject: outputUpdate,
    oldSecretPaths,
    updatedSecretPaths,
    esClient,
    secretHashes: outputUpdate.is_preconfigured ? secretHashes : undefined
  });
  return {
    outputUpdate: secretRes.updatedSoObject,
    secretReferences: secretRes.secretReferences,
    secretsToDelete: secretRes.secretsToDelete
  };
}
async function deleteOutputSecrets(opts) {
  const {
    output,
    esClient
  } = opts;
  const outputType = output.type;
  const outputSecretPaths = getOutputSecretPaths(outputType, output);
  await (0, _common.deleteSOSecrets)(esClient, outputSecretPaths);
}
function getOutputSecretReferences(output) {
  var _output$secrets, _output$secrets$ssl, _output$secrets2;
  const outputSecretPaths = [];
  if (typeof ((_output$secrets = output.secrets) === null || _output$secrets === void 0 ? void 0 : (_output$secrets$ssl = _output$secrets.ssl) === null || _output$secrets$ssl === void 0 ? void 0 : _output$secrets$ssl.key) === 'object') {
    outputSecretPaths.push({
      id: output.secrets.ssl.key.id
    });
  }
  if (output.type === 'kafka' && typeof (output === null || output === void 0 ? void 0 : (_output$secrets2 = output.secrets) === null || _output$secrets2 === void 0 ? void 0 : _output$secrets2.password) === 'object') {
    outputSecretPaths.push({
      id: output.secrets.password.id
    });
  }
  if (output.type === 'remote_elasticsearch') {
    var _output$secrets3;
    if (typeof (output === null || output === void 0 ? void 0 : (_output$secrets3 = output.secrets) === null || _output$secrets3 === void 0 ? void 0 : _output$secrets3.service_token) === 'object') {
      outputSecretPaths.push({
        id: output.secrets.service_token.id
      });
    }
  }
  return outputSecretPaths;
}
function getOutputSecretPaths(outputType, output) {
  var _output$secrets4, _output$secrets4$ssl;
  const outputSecretPaths = [];
  if (outputType === 'kafka') {
    var _kafkaOutput$secrets;
    const kafkaOutput = output;
    if (kafkaOutput !== null && kafkaOutput !== void 0 && (_kafkaOutput$secrets = kafkaOutput.secrets) !== null && _kafkaOutput$secrets !== void 0 && _kafkaOutput$secrets.password) {
      outputSecretPaths.push({
        path: 'secrets.password',
        value: kafkaOutput.secrets.password
      });
    }
  }
  if (outputType === 'remote_elasticsearch') {
    var _remoteESOutput$secre;
    const remoteESOutput = output;
    if ((_remoteESOutput$secre = remoteESOutput.secrets) !== null && _remoteESOutput$secre !== void 0 && _remoteESOutput$secre.service_token) {
      outputSecretPaths.push({
        path: 'secrets.service_token',
        value: remoteESOutput.secrets.service_token
      });
    }
  }

  // common to all outputs
  if (output !== null && output !== void 0 && (_output$secrets4 = output.secrets) !== null && _output$secrets4 !== void 0 && (_output$secrets4$ssl = _output$secrets4.ssl) !== null && _output$secrets4$ssl !== void 0 && _output$secrets4$ssl.key) {
    outputSecretPaths.push({
      path: 'secrets.ssl.key',
      value: output.secrets.ssl.key
    });
  }
  return outputSecretPaths;
}