"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getManagedId = getManagedId;
exports.insertOrReplaceFormulaColumn = insertOrReplaceFormulaColumn;
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _utils = require("../../../../../utils");
var _ = require("..");
var _math = require("./math");
var _document_field = require("../../../document_field");
var _validation = require("./validation");
var _util = require("./util");
var _formula = require("./formula");
var _layer_helpers = require("../../layer_helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/** @internal **/
function getManagedId(mainId, index) {
  return `${mainId}X${index}`;
}
function parseAndExtract(text, layer, columnId, indexPattern, operations, dateRange, label) {
  const parseResponse = (0, _validation.tryToParse)(text, operations);
  if ('error' in parseResponse) {
    return {
      extracted: [],
      isValid: false
    };
  }
  // before extracting the data run the validation task and throw if invalid
  const errors = (0, _validation.runASTValidation)(parseResponse.root, layer, indexPattern, operations, layer.columns[columnId], dateRange);
  if (errors.length) {
    return {
      extracted: [],
      isValid: false
    };
  }
  /*
    { name: 'add', args: [ { name: 'abc', args: [5] }, 5 ] }
    */
  const extracted = extractColumns(columnId, operations, parseResponse.root, layer, indexPattern, _i18n.i18n.translate('xpack.lens.indexPattern.formulaPartLabel', {
    defaultMessage: 'Part of {label}',
    values: {
      label: label || text
    }
  }), dateRange);
  return {
    extracted,
    isValid: true
  };
}
function extractColumns(idPrefix, operations, ast, layer, indexPattern, label, dateRange) {
  const columns = [];
  const {
    filter: globalFilter,
    reducedTimeRange: globalReducedTimeRange
  } = layer.columns[idPrefix];
  function parseNode(node) {
    if (typeof node === 'number' || node.type !== 'function') {
      // leaf node
      return node;
    }
    const nodeOperation = operations[node.name];
    if (!nodeOperation) {
      // it's a regular math node
      const consumedArgs = node.args.map(parseNode).filter(_utils.nonNullable);
      return {
        ...node,
        args: consumedArgs
      };
    }

    // split the args into types for better TS experience
    const {
      namedArguments,
      variables,
      functions
    } = (0, _util.groupArgsByType)(node.args);

    // operation node
    if (nodeOperation.input === 'field') {
      const [fieldName] = variables.filter(v => (0, _lodash.isObject)(v));
      // a validation task passed before executing this and checked already there's a field
      let field = fieldName ? indexPattern.getFieldByName(fieldName.value) : undefined;
      if (!(0, _validation.shouldHaveFieldArgument)(node) && !field) {
        field = _document_field.documentField;
      }
      const mappedParams = {
        ...(0, _util.mergeWithGlobalFilters)(nodeOperation, (0, _util.getOperationParams)(nodeOperation, namedArguments || []), globalFilter, globalReducedTimeRange),
        usedInMath: true
      };
      const newCol = nodeOperation.buildColumn({
        layer,
        indexPattern,
        // checked in the validation phase
        field: field
      }, mappedParams);
      const newColId = getManagedId(idPrefix, columns.length);
      newCol.customLabel = true;
      newCol.label = label;
      columns.push({
        column: newCol,
        location: node.location
      });
      // replace by new column id
      return newColId;
    }
    if (nodeOperation.input === 'fullReference') {
      const [referencedOp] = functions;
      const consumedParam = parseNode(referencedOp);
      const hasActualMathContent = typeof consumedParam !== 'string';
      if (hasActualMathContent) {
        const subNodeVariables = consumedParam ? (0, _util.findVariables)(consumedParam) : [];
        const mathColumn = _math.mathOperation.buildColumn({
          layer,
          indexPattern
        });
        mathColumn.references = subNodeVariables.map(({
          value
        }) => value);
        mathColumn.params.tinymathAst = consumedParam;
        columns.push({
          column: mathColumn
        });
        mathColumn.customLabel = true;
        mathColumn.label = label;
      }
      const mappedParams = (0, _util.mergeWithGlobalFilters)(nodeOperation, (0, _util.getOperationParams)(nodeOperation, namedArguments || []), globalFilter, globalReducedTimeRange);
      const newCol = nodeOperation.buildColumn({
        layer,
        indexPattern,
        referenceIds: [hasActualMathContent ? getManagedId(idPrefix, columns.length - 1) : consumedParam]
      }, mappedParams);
      const newColId = getManagedId(idPrefix, columns.length);
      newCol.customLabel = true;
      newCol.label = label;
      columns.push({
        column: newCol,
        location: node.location
      });
      // replace by new column id
      return newColId;
    }
    if (nodeOperation.input === 'managedReference' && nodeOperation.usedInMath) {
      const newCol = nodeOperation.buildColumn({
        layer,
        indexPattern
      });
      const newColId = getManagedId(idPrefix, columns.length);
      newCol.customLabel = true;
      newCol.label = label;
      columns.push({
        column: newCol,
        location: node.location
      });
      // replace by new column id
      return newColId;
    }
  }
  const root = parseNode(ast);
  if (root === undefined) {
    return [];
  }
  const topLevelMath = typeof root !== 'string';
  if (topLevelMath) {
    const variables = (0, _util.findVariables)(root);
    const mathColumn = _math.mathOperation.buildColumn({
      layer,
      indexPattern
    });
    mathColumn.references = variables.map(({
      value
    }) => value);
    mathColumn.params.tinymathAst = root;
    mathColumn.customLabel = true;
    mathColumn.label = label;
    columns.push({
      column: mathColumn
    });
  }
  return columns;
}
const getEmptyColumnsWithFormulaMeta = () => ({
  columns: {},
  meta: {
    locations: {}
  }
});
function generateFormulaColumns(id, column, layer, {
  indexPattern,
  operations = _.operationDefinitionMap,
  dateRange
}) {
  const {
    columns,
    meta
  } = getEmptyColumnsWithFormulaMeta();
  const formula = column.params.formula || '';
  const {
    extracted,
    isValid
  } = parseAndExtract(formula, layer, id, indexPattern, (0, _util.filterByVisibleOperation)(operations), dateRange, column.customLabel ? column.label : undefined);
  extracted.forEach(({
    column: extractedColumn,
    location
  }, index) => {
    const managedId = getManagedId(id, index);
    columns[managedId] = extractedColumn;
    if (location) {
      meta.locations[managedId] = location;
    }
  });
  columns[id] = {
    ...column,
    label: !column.customLabel ? formula !== null && formula !== void 0 ? formula : _i18n.i18n.translate('xpack.lens.indexPattern.formulaLabel', {
      defaultMessage: 'Formula'
    }) : column.label,
    references: !isValid ? [] : [getManagedId(id, extracted.length - 1)],
    params: {
      ...column.params,
      formula,
      isFormulaBroken: !isValid
    }
  };
  return {
    columns,
    meta
  };
}

/** @internal **/
function insertOrReplaceFormulaColumn(id, column, baseLayer, params) {
  const layer = {
    ...baseLayer,
    columns: {
      ...baseLayer.columns,
      [id]: {
        ...column
      }
    }
  };
  const {
    columns: updatedColumns,
    meta
  } = Object.entries(layer.columns).reduce((acc, [currentColumnId, currentColumn]) => {
    if (currentColumnId.startsWith(id)) {
      if (currentColumnId === id && (0, _formula.isFormulaIndexPatternColumn)(currentColumn)) {
        const formulaColumns = generateFormulaColumns(currentColumnId, currentColumn, layer, params);
        acc.columns = {
          ...acc.columns,
          ...formulaColumns.columns
        };
        acc.meta = {
          ...acc.meta,
          ...formulaColumns.meta
        };
      }
    } else {
      acc.columns[currentColumnId] = {
        ...currentColumn
      };
    }
    return acc;
  }, getEmptyColumnsWithFormulaMeta());
  return {
    layer: {
      ...layer,
      columns: updatedColumns,
      columnOrder: (0, _layer_helpers.getColumnOrder)({
        ...layer,
        columns: updatedColumns
      })
    },
    meta
  };
}