"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initRoutes = initRoutes;
var _configSchema = require("@kbn/config-schema");
var _fs = _interopRequireDefault(require("fs"));
var _path = _interopRequireDefault(require("path"));
var _constants = require("../common/constants");
var _get_index_pattern_settings = require("./lib/get_index_pattern_settings");
var _mvt_routes = require("./mvt/mvt_routes");
var _indexing_routes = require("./data_indexing/indexing_routes");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function initRoutes(coreSetup, logger) {
  const router = coreSetup.http.createRouter();
  router.versioned.get({
    path: `${_constants.FONTS_API_PATH}/{fontstack}/{range}`,
    access: 'internal',
    security: {
      authz: {
        enabled: false,
        reason: 'This route is opted out from authorization because it is only serving static files.'
      }
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _configSchema.schema.object({
          fontstack: _configSchema.schema.string(),
          range: _configSchema.schema.string()
        })
      }
    }
  }, (context, request, response) => {
    const range = _path.default.normalize(request.params.range);
    const rootPath = _path.default.resolve(__dirname, 'fonts', 'open_sans');
    const fontPath = _path.default.resolve(rootPath, `${range}.pbf`);
    return !fontPath.startsWith(rootPath) ? response.notFound() : new Promise(resolve => {
      _fs.default.readFile(fontPath, (error, data) => {
        if (error) {
          resolve(response.notFound());
        } else {
          resolve(response.ok({
            body: data
          }));
        }
      });
    });
  });
  router.versioned.get({
    path: _constants.INDEX_SETTINGS_API_PATH,
    access: 'internal',
    security: {
      authz: {
        enabled: false,
        reason: 'This route is opted out from authorization because permissions will be checked by elasticsearch.'
      }
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        query: _configSchema.schema.object({
          indexPatternTitle: _configSchema.schema.string()
        })
      }
    }
  }, async (context, request, response) => {
    const {
      query
    } = request;
    if (!query.indexPatternTitle) {
      logger.warn(`Required query parameter 'indexPatternTitle' not provided.`);
      return response.custom({
        body: `Required query parameter 'indexPatternTitle' not provided.`,
        statusCode: 400
      });
    }
    try {
      const coreContext = await context.core;
      const resp = await coreContext.elasticsearch.client.asCurrentUser.indices.getSettings({
        index: query.indexPatternTitle
      });
      const indexPatternSettings = (0, _get_index_pattern_settings.getIndexPatternSettings)(resp);
      return response.ok({
        body: indexPatternSettings
      });
    } catch (error) {
      logger.warn(`Cannot load index settings for data view '${query.indexPatternTitle}', error: ${error.message}.`);
      return response.custom({
        body: 'Error loading index settings',
        statusCode: 400
      });
    }
  });
  (0, _mvt_routes.initMVTRoutes)({
    router,
    logger,
    getCore: async () => {
      const [core] = await coreSetup.getStartServices();
      return core;
    }
  });
  (0, _indexing_routes.initIndexingRoutes)({
    router,
    logger,
    getDataPlugin: async () => {
      const [, {
        data
      }] = await coreSetup.getStartServices();
      return data;
    }
  });
}