"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PainlessCompletionAdapter = void 0;
var _monaco_imports = require("../../monaco_imports");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const getCompletionKind = kind => {
  const monacoItemKind = _monaco_imports.monaco.languages.CompletionItemKind;
  switch (kind) {
    case 'type':
      return monacoItemKind.Interface;
    case 'class':
      return monacoItemKind.Class;
    case 'method':
      return monacoItemKind.Method;
    case 'constructor':
      return monacoItemKind.Constructor;
    case 'property':
      return monacoItemKind.Property;
    case 'keyword':
      return monacoItemKind.Keyword;
    case 'field':
      return monacoItemKind.Field;
    default:
      return monacoItemKind.Text;
  }
};
class PainlessCompletionAdapter {
  constructor(worker, editorStateService) {
    this.worker = worker;
    this.editorStateService = editorStateService;
  }
  get triggerCharacters() {
    return ['.', `'`];
  }
  async provideCompletionItems(model, position) {
    // Active line characters
    const currentLineChars = model.getValueInRange({
      startLineNumber: position.lineNumber,
      startColumn: 0,
      endLineNumber: position.lineNumber,
      endColumn: position.column
    });
    const worker = await this.worker(model.uri);
    const {
      context,
      fields
    } = this.editorStateService.getState();
    const autocompleteInfo = await worker.provideAutocompleteSuggestions(currentLineChars, context, fields);
    const wordInfo = model.getWordUntilPosition(position);
    const wordRange = {
      startLineNumber: position.lineNumber,
      endLineNumber: position.lineNumber,
      startColumn: wordInfo.startColumn,
      endColumn: wordInfo.endColumn
    };
    const suggestions = autocompleteInfo.suggestions.map(({
      label,
      insertText,
      documentation,
      kind,
      insertTextAsSnippet
    }) => {
      return {
        label,
        insertText,
        documentation,
        range: wordRange,
        kind: getCompletionKind(kind),
        insertTextRules: insertTextAsSnippet ? _monaco_imports.monaco.languages.CompletionItemInsertTextRule.InsertAsSnippet : undefined
      };
    });
    return {
      incomplete: autocompleteInfo.isIncomplete,
      suggestions
    };
  }
}
exports.PainlessCompletionAdapter = PainlessCompletionAdapter;