"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.formatAction = void 0;
exports.isAutocorrectQueryAction = isAutocorrectQueryAction;
exports.isExecuteQueryAction = isExecuteQueryAction;
exports.isGenerateQueryAction = isGenerateQueryAction;
exports.isRequestDocumentationAction = isRequestDocumentationAction;
var _messages = require("../../langchain/messages");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function isRequestDocumentationAction(action) {
  return action.type === 'request_documentation';
}
function isGenerateQueryAction(action) {
  return action.type === 'generate_query';
}
function isAutocorrectQueryAction(action) {
  return action.type === 'autocorrect_query';
}
function isExecuteQueryAction(action) {
  return action.type === 'execute_query';
}

/**
 * Format an action into a couple of [ai, user] messages to be used in prompts.
 */
const formatAction = (action, withoutToolCalls = true) => {
  // Important notice: Claude is *very* stupid with tool configuration
  // and will be fine calling tools that are not available, just based on previous tool calls
  // which means we can't represent the action history as a tool call list
  // and are forced to similar a conversation instead.
  // yes, this is sub-optimal, but this is how Claude behaves.

  const toolCallId = (0, _messages.generateFakeToolCallId)();
  switch (action.type) {
    case 'generate_query':
      return [(0, _messages.createAIMessage)(action.response), (0, _messages.createUserMessage)(action.success ? `Thank you` : `I don't see any query in your response, can you please try again? Don't forget to wrap your query with \`\`\`esql[query]\`\`\``)];
    case 'autocorrect_query':
      if (!action.wasCorrected) {
        return [];
      }
      return withoutToolCalls ? [(0, _messages.createAIMessage)('Now you can execute the query'), (0, _messages.createUserMessage)(`I ran the query through autocorrect and the corrected query was:

 \`\`\`esql
 "${action.output}"
 \`\`\``)] : [(0, _messages.createToolCallMessage)({
        toolCallId,
        toolName: 'correct_query',
        args: {
          query: action.input
        }
      }), (0, _messages.createToolResultMessage)({
        toolCallId,
        content: {
          hasMistakes: action.wasCorrected,
          output: action.output
        }
      })];
    case 'execute_query':
      if (action.success) {
        return [];
      }
      return withoutToolCalls ? [(0, _messages.createAIMessage)('Now you can execute the query'), (0, _messages.createUserMessage)(`I tried executing the query and got the following error:

\`\`\`
${action.error}
\`\`\`

Can you fix the query?`)] : [(0, _messages.createToolCallMessage)({
        toolCallId,
        toolName: 'execute_query',
        args: {
          query: action.query
        }
      }), (0, _messages.createToolResultMessage)({
        toolCallId,
        content: {
          success: action.success,
          error: action.error
        }
      })];
    case 'request_documentation':
      // always use tool call format for this action, to stay closer to the original flow
      // also Claude doesn't seem to care about requesting more doc.
      return [(0, _messages.createToolCallMessage)({
        toolCallId,
        toolName: 'request_documentation',
        args: {
          keywords: action.requestedKeywords
        }
      }), (0, _messages.createToolResultMessage)({
        toolCallId,
        content: {
          documentation: action.fetchedDoc
        }
      })];
  }
};
exports.formatAction = formatAction;