"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.KibanaA2AAdapter = void 0;
var _server = require("@a2a-js/sdk/server");
var _create_agent_card = require("./create_agent_card");
var _kibana_agent_executor = require("./kibana_agent_executor");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Kibana adapter for the A2A SDK
 */
class KibanaA2AAdapter {
  constructor(logger, getInternalServices, getBaseUrl) {
    this.logger = logger;
    this.getInternalServices = getInternalServices;
    this.getBaseUrl = getBaseUrl;
  }

  /**
   * Create A2A components for a specific agent and request
   */
  async createA2AComponents(kibanaRequest, agentId) {
    // Get agent and create agent card
    const {
      agents,
      tools
    } = this.getInternalServices();
    const agentRegistry = await agents.getRegistry({
      request: kibanaRequest
    });
    const agent = await agentRegistry.get(agentId);
    const agentCard = await (0, _create_agent_card.createAgentCard)({
      agent,
      baseUrl: await this.getBaseUrl(kibanaRequest),
      toolsService: tools,
      request: kibanaRequest
    });

    // Kibana load balancing lacks session affinity, user requests may hit different nodes.
    // This task store is unused but required by the A2A SDK.
    const taskStore = new _server.InMemoryTaskStore();
    const agentExecutor = new _kibana_agent_executor.KibanaAgentExecutor(this.logger, this.getInternalServices, kibanaRequest, agentId);
    const requestHandler = new _server.DefaultRequestHandler(agentCard, taskStore, agentExecutor);
    const jsonRpcHandler = new _server.JsonRpcTransportHandler(requestHandler);
    return {
      requestHandler,
      jsonRpcHandler
    };
  }

  /**
   * Handle agent card requests
   */
  async handleAgentCardRequest(req, res, agentId) {
    try {
      const {
        requestHandler
      } = await this.createA2AComponents(req, agentId);
      const agentCard = await requestHandler.getAgentCard();
      return res.ok({
        headers: {
          'Content-Type': 'application/json',
          'Cache-Control': 'public, max-age=3600'
        },
        body: agentCard
      });
    } catch (error) {
      this.logger.error(`A2A: Failed to serve agent card for ${agentId}: ${error}`);
      return res.customError({
        statusCode: 500,
        body: {
          message: `Failed to serve agent card: ${error}`
        }
      });
    }
  }

  /**
   * Handle A2A JSON-RPC requests
   */
  async handleA2ARequest(req, res, agentId) {
    try {
      this.logger.debug(`A2A: Processing request for agent ${agentId}`);

      // Validate content type
      const contentType = req.headers['content-type'];
      if (!(contentType !== null && contentType !== void 0 && contentType.includes('application/json'))) {
        return res.badRequest({
          body: {
            message: 'Content-Type must be application/json'
          }
        });
      }

      // Process request through A2A SDK
      const {
        jsonRpcHandler
      } = await this.createA2AComponents(req, agentId);
      const result = await jsonRpcHandler.handle(req.body);
      return res.ok({
        headers: {
          'Content-Type': 'application/json'
        },
        body: result
      });
    } catch (error) {
      this.logger.error(`A2A: Request failed for agent ${agentId}: ${error}`);
      if (error instanceof _server.A2AError) {
        return res.badRequest({
          body: error.toJSONRPCError()
        });
      }
      return res.customError({
        statusCode: 500,
        body: {
          message: `Internal server error: ${error}`
        }
      });
    }
  }

  /**
   * Handle unsupported methods
   */
  async handleUnsupportedRequest(req, res) {
    this.logger.warn(`A2A: Unsupported method: ${req.route.method}`);
    return res.customError({
      statusCode: 405,
      body: {
        message: 'Method not allowed'
      }
    });
  }
}
exports.KibanaA2AAdapter = KibanaA2AAdapter;