"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.populateIndex = void 0;
var _productDocCommon = require("@kbn/product-doc-common");
var _consts = require("../../../../common/consts");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const populateIndex = async ({
  esClient,
  indexName,
  manifestVersion,
  archive,
  log,
  inferenceId = _consts.internalElserInferenceId
}) => {
  log.debug(`Starting populating index ${indexName}`);
  const legacySemanticText = (0, _utils.isLegacySemanticTextVersion)(manifestVersion);
  const contentEntries = archive.getEntryPaths().filter(_productDocCommon.isArtifactContentFilePath);
  for (let i = 0; i < contentEntries.length; i++) {
    const entryPath = contentEntries[i];
    log.debug(`Indexing content for entry ${entryPath}`);
    const contentBuffer = await archive.getEntryContent(entryPath);
    await indexContentFile({
      indexName,
      esClient,
      contentBuffer,
      legacySemanticText,
      inferenceId
    });
  }
  log.debug(`Done populating index ${indexName}`);
};
exports.populateIndex = populateIndex;
const indexContentFile = async ({
  indexName,
  contentBuffer,
  esClient,
  legacySemanticText,
  elserInferenceId = _consts.internalElserInferenceId,
  inferenceId
}) => {
  const fileContent = contentBuffer.toString('utf-8');
  const lines = fileContent.split('\n');
  const documents = lines.map(line => line.trim()).filter(line => line.length > 0).map(line => {
    return JSON.parse(line);
  }).map(doc => rewriteInferenceId({
    document: doc,
    inferenceId: inferenceId !== null && inferenceId !== void 0 ? inferenceId : elserInferenceId,
    legacySemanticText
  }));
  const operations = documents.reduce((ops, document) => {
    ops.push(...[{
      index: {
        _index: indexName
      }
    }, document]);
    return ops;
  }, []);
  const response = await esClient.bulk({
    refresh: false,
    operations
  });
  if (response.errors) {
    var _response$items$find$, _response$items$find, _response$items$find$2;
    const error = (_response$items$find$ = (_response$items$find = response.items.find(item => {
      var _item$index;
      return (_item$index = item.index) === null || _item$index === void 0 ? void 0 : _item$index.error;
    })) === null || _response$items$find === void 0 ? void 0 : (_response$items$find$2 = _response$items$find.index) === null || _response$items$find$2 === void 0 ? void 0 : _response$items$find$2.error) !== null && _response$items$find$ !== void 0 ? _response$items$find$ : 'unknown error';
    throw new Error(`Error indexing documents: ${JSON.stringify(error)}`);
  }
};
const rewriteInferenceId = ({
  document,
  inferenceId,
  legacySemanticText
}) => {
  const semanticFieldsRoot = legacySemanticText ? document : document._inference_fields;
  // we don't need to handle nested fields, we don't have any and won't.
  Object.values(semanticFieldsRoot !== null && semanticFieldsRoot !== void 0 ? semanticFieldsRoot : {}).forEach(field => {
    if (field.inference) {
      field.inference.inference_id = inferenceId;
    }
  });
  return document;
};