"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.is = exports.convertQuery = exports.convertPathParameters = exports.convert = void 0;
var _zod = require("@kbn/zod");
var _zodHelpers = require("@kbn/zod-helpers");
var _zodToJsonSchema = _interopRequireDefault(require("zod-to-json-schema"));
var _common = require("../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// Adapted from from https://github.com/jlalmes/trpc-openapi/blob/aea45441af785518df35c2bc173ae2ea6271e489/src/utils/zod.ts#L1

const createError = message => {
  return new Error(`[Zod converter] ${message}`);
};
function assertInstanceOfZodType(schema) {
  if (!(0, _zod.isZod)(schema)) {
    throw createError('Expected schema to be an instance of Zod');
  }
}
const instanceofZodTypeKind = (type, zodTypeKind) => {
  var _type$_def;
  return (type === null || type === void 0 ? void 0 : (_type$_def = type._def) === null || _type$_def === void 0 ? void 0 : _type$_def.typeName) === zodTypeKind;
};
const instanceofZodTypeObject = type => {
  return instanceofZodTypeKind(type, _zod.z.ZodFirstPartyTypeKind.ZodObject);
};
const instanceofZodTypeLikeVoid = type => {
  return instanceofZodTypeKind(type, _zod.z.ZodFirstPartyTypeKind.ZodVoid) || instanceofZodTypeKind(type, _zod.z.ZodFirstPartyTypeKind.ZodUndefined) || instanceofZodTypeKind(type, _zod.z.ZodFirstPartyTypeKind.ZodNever);
};
const unwrapZodLazy = type => {
  if (instanceofZodTypeKind(type, _zod.z.ZodFirstPartyTypeKind.ZodLazy)) {
    return unwrapZodLazy(type._def.getter());
  }
  return type;
};
const unwrapZodOptionalDefault = type => {
  let description; // To track the outer description if exists
  let defaultValue;
  let isOptional = false;
  let innerType = type;
  while (instanceofZodTypeKind(innerType, _zod.z.ZodFirstPartyTypeKind.ZodOptional) || instanceofZodTypeKind(innerType, _zod.z.ZodFirstPartyTypeKind.ZodDefault)) {
    if (instanceofZodTypeKind(innerType, _zod.z.ZodFirstPartyTypeKind.ZodOptional)) {
      isOptional = innerType.isOptional();
      description = !description ? innerType.description : description;
      innerType = innerType.unwrap();
    }
    if (instanceofZodTypeKind(innerType, _zod.z.ZodFirstPartyTypeKind.ZodDefault)) {
      defaultValue = innerType._def.defaultValue();
      description = !description ? innerType.description : description;
      innerType = innerType.removeDefault();
    }
  }
  return {
    description,
    defaultValue,
    isOptional,
    innerType
  };
};
const unwrapZodType = (type, unwrapPreprocess) => {
  if (instanceofZodTypeKind(type, _zod.z.ZodFirstPartyTypeKind.ZodLazy)) {
    return unwrapZodType(unwrapZodLazy(type), unwrapPreprocess);
  }
  if (instanceofZodTypeKind(type, _zod.z.ZodFirstPartyTypeKind.ZodOptional) || instanceofZodTypeKind(type, _zod.z.ZodFirstPartyTypeKind.ZodDefault)) {
    const {
      innerType
    } = unwrapZodOptionalDefault(type);
    return unwrapZodType(innerType, unwrapPreprocess);
  }
  if (instanceofZodTypeKind(type, _zod.z.ZodFirstPartyTypeKind.ZodEffects)) {
    if (type._def.effect.type === 'refinement') {
      return unwrapZodType(type._def.schema, unwrapPreprocess);
    }
    if (type._def.effect.type === 'transform') {
      return unwrapZodType(type._def.schema, unwrapPreprocess);
    }
    if (unwrapPreprocess && type._def.effect.type === 'preprocess') {
      return unwrapZodType(type._def.schema, unwrapPreprocess);
    }
  }
  return type;
};
const zodSupportsCoerce = 'coerce' in _zod.z;
const instanceofZodTypeCoercible = _type => {
  const type = unwrapZodType(_type, false);
  return instanceofZodTypeKind(type, _zod.z.ZodFirstPartyTypeKind.ZodNumber) || instanceofZodTypeKind(type, _zod.z.ZodFirstPartyTypeKind.ZodBoolean) || instanceofZodTypeKind(type, _zod.z.ZodFirstPartyTypeKind.ZodBigInt) || instanceofZodTypeKind(type, _zod.z.ZodFirstPartyTypeKind.ZodDate);
};
const instanceofZodTypeLikeString = (_type, allowMixedUnion) => {
  const type = unwrapZodType(_type, false);
  if (instanceofZodTypeKind(type, _zod.z.ZodFirstPartyTypeKind.ZodEffects)) {
    if (type._def.effect.type === 'preprocess') {
      return true;
    }
  }
  if (instanceofZodTypeKind(type, _zod.z.ZodFirstPartyTypeKind.ZodUnion)) {
    return !type._def.options.some(option => !instanceofZodTypeLikeString(option, allowMixedUnion) && !(allowMixedUnion && instanceofZodTypeCoercible(option)));
  }
  if (instanceofZodTypeKind(type, _zod.z.ZodFirstPartyTypeKind.ZodArray)) {
    return instanceofZodTypeLikeString(type._def.type, allowMixedUnion);
  }
  if (instanceofZodTypeKind(type, _zod.z.ZodFirstPartyTypeKind.ZodIntersection)) {
    return instanceofZodTypeLikeString(type._def.left, allowMixedUnion) && instanceofZodTypeLikeString(type._def.right, allowMixedUnion);
  }
  if (instanceofZodTypeKind(type, _zod.z.ZodFirstPartyTypeKind.ZodLiteral)) {
    return typeof type._def.value === 'string';
  }
  if (instanceofZodTypeKind(type, _zod.z.ZodFirstPartyTypeKind.ZodEnum)) {
    return true;
  }
  if (instanceofZodTypeKind(type, _zod.z.ZodFirstPartyTypeKind.ZodNativeEnum)) {
    return !Object.values(type._def.values).some(value => typeof value === 'number');
  }
  return instanceofZodTypeKind(type, _zod.z.ZodFirstPartyTypeKind.ZodString);
};
const convertObjectMembersToParameterObjects = (shape, isPathParameter = false, knownParameters = {}) => {
  return Object.entries(shape).map(([shapeKey, subShape]) => {
    var _knownParameters$shap;
    const typeWithoutLazy = unwrapZodLazy(subShape);
    const {
      description: outerDescription,
      isOptional,
      defaultValue,
      innerType: typeWithoutOptionalDefault
    } = unwrapZodOptionalDefault(typeWithoutLazy);

    // Except for path parameters, OpenAPI supports mixed unions with `anyOf` e.g. for query parameters
    if (!instanceofZodTypeLikeString(typeWithoutOptionalDefault, !isPathParameter)) {
      if (zodSupportsCoerce) {
        if (!instanceofZodTypeCoercible(typeWithoutOptionalDefault)) {
          throw createError(`Input parser key: "${shapeKey}" must be ZodString, ZodNumber, ZodBoolean, ZodBigInt or ZodDate`);
        }
      } else {
        throw createError(`Input parser key: "${shapeKey}" must be ZodString`);
      }
    }
    const {
      schema: {
        description: schemaDescription,
        ...openApiSchemaObject
      }
    } = convert(typeWithoutOptionalDefault);
    if (typeof defaultValue !== 'undefined') {
      openApiSchemaObject.default = defaultValue;
    }
    return {
      name: shapeKey,
      in: isPathParameter ? 'path' : 'query',
      required: isPathParameter ? !((_knownParameters$shap = knownParameters[shapeKey]) !== null && _knownParameters$shap !== void 0 && _knownParameters$shap.optional) : !isOptional,
      schema: openApiSchemaObject,
      description: outerDescription || schemaDescription
    };
  });
};

// Returns a z.ZodRawShape to passes through all known parameters with z.any
const getPassThroughShape = (knownParameters, isPathParameter = false) => {
  const passThroughShape = {};
  for (const [key, {
    optional
  }] of Object.entries(knownParameters)) {
    passThroughShape[key] = optional && !isPathParameter ? _zod.z.string().optional() : _zod.z.string();
  }
  return passThroughShape;
};
const convertQuery = schema => {
  assertInstanceOfZodType(schema);
  const unwrappedSchema = unwrapZodType(schema, true);
  if ((0, _zodHelpers.isPassThroughAny)(unwrappedSchema)) {
    return {
      query: convertObjectMembersToParameterObjects(getPassThroughShape({}, false), true),
      shared: {}
    };
  }
  if (!instanceofZodTypeObject(unwrappedSchema)) {
    throw createError('Query schema must be an _object_ schema validator!');
  }
  const shape = unwrappedSchema.shape;
  return {
    query: convertObjectMembersToParameterObjects(shape, false),
    shared: {}
  };
};
exports.convertQuery = convertQuery;
const convertPathParameters = (schema, knownParameters) => {
  assertInstanceOfZodType(schema);
  const unwrappedSchema = unwrapZodType(schema, true);
  const paramKeys = Object.keys(knownParameters);
  const paramsCount = paramKeys.length;
  if (paramsCount === 0 && instanceofZodTypeLikeVoid(unwrappedSchema)) {
    return {
      params: [],
      shared: {}
    };
  }
  if ((0, _zodHelpers.isPassThroughAny)(unwrappedSchema)) {
    return {
      params: convertObjectMembersToParameterObjects(getPassThroughShape(knownParameters, true), true),
      shared: {}
    };
  }
  if (!instanceofZodTypeObject(unwrappedSchema)) {
    throw createError('Parameters schema must be an _object_ schema validator!');
  }
  const shape = unwrappedSchema.shape;
  const schemaKeys = Object.keys(shape);
  (0, _common.validatePathParameters)(paramKeys, schemaKeys);
  return {
    params: convertObjectMembersToParameterObjects(shape, true),
    shared: {}
  };
};
exports.convertPathParameters = convertPathParameters;
const convert = schema => {
  return {
    shared: {},
    schema: (0, _zodToJsonSchema.default)(schema, {
      target: 'openApi3',
      $refStrategy: 'none'
    })
  };
};
exports.convert = convert;
const is = exports.is = _zod.isZod;