"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SavedObjectsManager = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/core/server");
var _stream = require("stream");
var _saved_objects = require("../../saved_objects/saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class SavedObjectsManager {
  constructor({
    logger
  }) {
    (0, _defineProperty2.default)(this, "log", void 0);
    this.log = logger;
  }
  async importSavedObjects(soImporter, sampleType) {
    const savedObjects = (0, _saved_objects.getSavedObjects)(sampleType);
    const readStream = _stream.Readable.from(savedObjects);
    const {
      errors = []
    } = await soImporter.import({
      readStream,
      overwrite: true,
      createNewCopies: false
    });
    if (errors.length > 0) {
      const errMsg = `Errors while loading saved objects. Errors: ${JSON.stringify(errors.map(({
        type,
        id,
        error
      }) => ({
        type,
        id,
        error
      })))}`;
      this.log.warn(errMsg);
      throw new Error(errMsg);
    }
    const dashboardId = (0, _saved_objects.getDashboardId)(sampleType);
    return {
      savedObjects,
      dashboardId
    };
  }
  async deleteSavedObjects(soClient, sampleType) {
    const savedObjects = (0, _saved_objects.getSavedObjects)(sampleType);
    let deletedCount = 0;
    const deletePromises = savedObjects.map(async ({
      type,
      id
    }) => {
      try {
        await soClient.delete(type, id);
        deletedCount++;
      } catch (err) {
        // if the object doesn't exist, ignore the error and proceed
        if (_server.SavedObjectsErrorHelpers.isNotFoundError(err)) {
          this.log.debug(`Saved object ${type}:${id} not found, skipping deletion`);
          return;
        }
        this.log.error(`Failed to delete saved object ${type}:${id}: ${err.message}`);
        throw err;
      }
    });
    try {
      await Promise.all(deletePromises);
    } catch (err) {
      throw new Error(`Unable to delete sample dataset saved objects, error: ${err.message}`);
    }
    return deletedCount;
  }
  async getDashboardId(soClient, sampleType) {
    const id = (0, _saved_objects.getDashboardId)(sampleType);
    if (!id) {
      return undefined;
    }
    try {
      await soClient.get('dashboard', id);
      return id;
    } catch (err) {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(err)) {
        this.log.debug(`Dashboard ${id} not found in saved objects`);
        return undefined;
      }
      this.log.error(`Failed to get dashboard id for sample type ${sampleType}: ${err.message}`);
      throw err;
    }
  }
}
exports.SavedObjectsManager = SavedObjectsManager;