"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.toValidIndexName = exports.generateConnectorName = void 0;
var _uuid = require("uuid");
var _lodash = require("lodash");
var _to_alphanumeric = require("../utils/to_alphanumeric");
var _exists_index = require("./exists_index");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const GENERATE_INDEX_NAME_ERROR = 'generate_index_name_error';
const toValidIndexName = str => {
  if (!str || str.trim() === '') {
    return 'index';
  }

  // Start with kebabCase to handle most transformations
  let result = (0, _lodash.kebabCase)(str);

  // Additional processing for ES index name requirements
  result = result
  // ES doesn't allow \, /, *, ?, ", <, >, |, comma, #, :
  .replace(/[\\/*?"<>|,#:]/g, '-')
  // Cannot start with -, _, +
  .replace(/^[-_+]/, '');

  // Remove trailing hyphens
  while (result.endsWith('-')) {
    result = result.slice(0, -1);
  }
  return result;
};
exports.toValidIndexName = toValidIndexName;
const generateConnectorName = async (client, connectorType, isNative, userConnectorName) => {
  const prefix = (0, _to_alphanumeric.toAlphanumeric)(connectorType);
  if (!prefix || prefix.length === 0) {
    throw new Error('Connector type or connectorName is required');
  }
  const nativePrefix = isNative ? _constants.MANAGED_CONNECTOR_INDEX_PREFIX : '';

  // Handle user-provided connector name
  if (userConnectorName) {
    // Keep original connector name, but sanitize it for index name
    const sanitizedName = toValidIndexName(userConnectorName);

    // First try with the sanitized name directly
    let indexName = `${nativePrefix}connector-${sanitizedName}`;
    const baseNameExists = await (0, _exists_index.indexOrAliasExists)(client, indexName);
    if (!baseNameExists) {
      return {
        connectorName: userConnectorName,
        // Keep original connector name
        indexName
      };
    }

    // If base name exists, try with random suffixes
    for (let i = 0; i < 20; i++) {
      const uniqueSuffix = (0, _uuid.v4)().split('-')[1].slice(0, 4);
      indexName = `${nativePrefix}connector-${sanitizedName}-${uniqueSuffix}`;
      const exists = await (0, _exists_index.indexOrAliasExists)(client, indexName);
      if (!exists) {
        return {
          connectorName: userConnectorName,
          // Keep original connector name
          indexName
        };
      }
    }
  } else {
    // Auto-generate a connector name
    for (let i = 0; i < 20; i++) {
      const uniqueSuffix = (0, _uuid.v4)().split('-')[1].slice(0, 4);
      const connectorName = `${toValidIndexName(prefix)}-${uniqueSuffix}`;
      const indexName = `${nativePrefix}connector-${connectorName}`;
      const exists = await (0, _exists_index.indexOrAliasExists)(client, indexName);
      if (!exists) {
        return {
          connectorName,
          indexName
        };
      }
    }
  }
  throw new Error(GENERATE_INDEX_NAME_ERROR);
};
exports.generateConnectorName = generateConnectorName;