"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TimeDuration = void 0;
var _zod = require("@kbn/zod");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isTimeSafe = time => time >= 1 && Number.isSafeInteger(time);

/**
 * Types the TimeDuration as:
 *   - A string that is not empty, and composed of a positive integer greater than 0 followed by a unit of time
 *   - in the format {safe_integer}{timeUnit}, e.g. "30s", "1m", "2h", "7d"
 *
 * Example usage:
 * ```
 * const schedule: RuleSchedule = {
 *   interval: TimeDuration({ allowedUnits: ['s', 'm', 'h'] }).parse('3h'),
 * };
 * ```
 */
const TimeDuration = ({
  allowedUnits
}) => {
  return _zod.z.string().refine(input => {
    if (input.trim() === '') return false;
    try {
      const inputLength = input.length;
      const time = Number(input.trim().substring(0, inputLength - 1));
      const unit = input.trim().at(-1);
      return isTimeSafe(time) && allowedUnits.includes(unit);
    } catch (error) {
      return false;
    }
  }, {
    message: 'Invalid time duration format. Must be a string that is not empty, and composed of a positive integer greater than 0 followed by a unit of time in the format {safe_integer}{timeUnit}, e.g. "30s", "1m", "2h", "7d"'
  });
};
exports.TimeDuration = TimeDuration;