"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getFieldArray = exports.getField = exports.getEventTitle = exports.getAlertTitle = exports.EVENT_CATEGORY_TO_FIELD = void 0;
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Helper function to retrieve a field's value (used in combination with the custom hook useGetFieldsData (x-pack/solutions/security/plugins/security_solution/public/flyout/document_details/shared/hooks/use_get_fields_data.ts)
 * @param field type unknown or unknown[]
 * @param emptyValue optional parameter to return if field is incorrect
 * @return the field's value, or null/emptyValue
 */
const getField = (field, emptyValue) => {
  if (typeof field === 'string') {
    return field;
  } else if (Array.isArray(field) && field.length > 0 && typeof field[0] === 'string') {
    return field[0];
  }
  return emptyValue !== null && emptyValue !== void 0 ? emptyValue : null;
};

/**
 * Helper function to retrieve a field's value in an array
 * @param field type unknown or unknown[]
 * @return the field's value in an array
 */
exports.getField = getField;
const getFieldArray = field => {
  if (typeof field === 'string') {
    return [field];
  } else if (Array.isArray(field) && field.length > 0) {
    return field;
  }
  return [];
};

// mapping of event category to the field displayed as title
exports.getFieldArray = getFieldArray;
const EVENT_CATEGORY_TO_FIELD = exports.EVENT_CATEGORY_TO_FIELD = {
  authentication: 'user.name',
  configuration: '',
  database: '',
  driver: '',
  email: '',
  file: 'file.name',
  host: 'host.name',
  iam: '',
  intrusion_detection: '',
  malware: '',
  network: '',
  package: '',
  process: 'process.name',
  registry: '',
  session: '',
  threat: '',
  vulnerability: '',
  web: ''
};

/**
 * Helper function to retrieve the alert title
 */
const getAlertTitle = ({
  ruleName
}) => {
  const defaultAlertTitle = _i18n.i18n.translate('xpack.securitySolution.flyout.right.header.headerTitle', {
    defaultMessage: 'Document details'
  });
  return ruleName !== null && ruleName !== void 0 ? ruleName : defaultAlertTitle;
};

/**
 * Helper function to retrieve the event title
 */
exports.getAlertTitle = getAlertTitle;
const getEventTitle = ({
  eventKind,
  eventCategory,
  getFieldsData
}) => {
  const defaultTitle = _i18n.i18n.translate('xpack.securitySolution.flyout.title.eventTitle', {
    defaultMessage: `Event details`
  });
  if (eventKind === 'event' && eventCategory) {
    var _getField;
    const fieldName = EVENT_CATEGORY_TO_FIELD[eventCategory];
    return (_getField = getField(getFieldsData(fieldName))) !== null && _getField !== void 0 ? _getField : defaultTitle;
  }
  if (eventKind === 'alert') {
    return _i18n.i18n.translate('xpack.securitySolution.flyout.title.alertEventTitle', {
      defaultMessage: 'External alert details'
    });
  }
  return eventKind ? _i18n.i18n.translate('xpack.securitySolution.flyout.title.otherEventTitle', {
    defaultMessage: '{eventKind} details',
    values: {
      eventKind: (0, _lodash.startCase)(eventKind)
    }
  }) : defaultTitle;
};
exports.getEventTitle = getEventTitle;