"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useRenderDelay = exports.usePendingActionsOptions = exports.useGenericErrorToast = exports.useFocusManagement = exports.useBaseSelectorState = exports.useBaseSelectorHandlers = void 0;
var _react = require("react");
var _securitysolutionHookUtils = require("@kbn/securitysolution-hook-utils");
var _momentTimezone = _interopRequireDefault(require("moment-timezone"));
var _constants = require("../../../../../common/endpoint/service/response_actions/constants");
var _kibana = require("../../../../common/lib/kibana");
var _translations = require("../../../common/translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Generic error toast hook that handles both custom scripts and pending actions errors
 */
const useGenericErrorToast = (error, notifications, errorTitlePrefix) => {
  (0, _react.useEffect)(() => {
    if (error) {
      let code = 'Error';
      let message;
      const err = error;
      if (err !== null && err !== void 0 && err.body && typeof err.body === 'object' && 'message' in err.body) {
        var _errorBody$statusCode;
        const errorBody = err.body;
        message = errorBody.message;
        code = String((_errorBody$statusCode = errorBody.statusCode) !== null && _errorBody$statusCode !== void 0 ? _errorBody$statusCode : code);
      } else {
        message = (err === null || err === void 0 ? void 0 : err.message) || String(err);
      }
      if (message) {
        notifications.toasts.addDanger({
          title: (0, _translations.getGenericErrorMessage)(errorTitlePrefix || '', code),
          text: message
        });
      }
    }
  }, [error, notifications, errorTitlePrefix]);
};

/**
 * Hook to manage base selector state
 */
exports.useGenericErrorToast = useGenericErrorToast;
const useBaseSelectorState = (store, value) => {
  return (0, _react.useMemo)(() => {
    return store !== null && store !== void 0 ? store : {
      isPopoverOpen: !value
    };
  }, [store, value]);
};

/**
 * Hook to create base selector handlers
 */
exports.useBaseSelectorState = useBaseSelectorState;
const useBaseSelectorHandlers = (state, onChange, value, valueText) => {
  const setIsPopoverOpen = (0, _react.useCallback)(newValue => {
    onChange({
      value,
      valueText,
      store: {
        ...state,
        isPopoverOpen: newValue
      }
    });
  }, [onChange, state, value, valueText]);
  const handleOpenPopover = (0, _react.useCallback)(() => {
    setIsPopoverOpen(true);
  }, [setIsPopoverOpen]);
  const handleClosePopover = (0, _react.useCallback)(() => {
    setIsPopoverOpen(false);
  }, [setIsPopoverOpen]);
  return {
    setIsPopoverOpen,
    handleOpenPopover,
    handleClosePopover
  };
};

/**
 * Hook to manage render delay state (handles race condition with parent input)
 */
exports.useBaseSelectorHandlers = useBaseSelectorHandlers;
const useRenderDelay = () => {
  const [isAwaitingRenderDelay, setIsAwaitingRenderDelay] = (0, _react.useState)(true);
  (0, _react.useEffect)(() => {
    const timer = setTimeout(() => {
      setIsAwaitingRenderDelay(false);
    }, 0);
    return () => clearTimeout(timer);
  }, []);
  return isAwaitingRenderDelay;
};

/**
 * Hook to handle focus management when popover closes
 */
exports.useRenderDelay = useRenderDelay;
const useFocusManagement = (isPopoverOpen, requestFocus) => {
  const getIsMounted = (0, _securitysolutionHookUtils.useIsMounted)();
  (0, _react.useEffect)(() => {
    if (!isPopoverOpen && requestFocus) {
      // Use setTimeout to ensure focus happens after the popover closes
      setTimeout(() => {
        if (getIsMounted() && requestFocus) {
          requestFocus();
        }
      }, 0);
    }
  }, [isPopoverOpen, requestFocus, getIsMounted]);
};

/**
 * Format timestamp using user's preferred date format and timezone settings
 */
exports.useFocusManagement = useFocusManagement;
const formatTimestamp = (timestamp, dateFormat, timeZone) => {
  return _momentTimezone.default.tz(timestamp, timeZone).format(dateFormat);
};

/**
 * Hook to transform pending actions response to selectable options with user's preferred date formatting
 */
const usePendingActionsOptions = ({
  response,
  selectedValue,
  privilegeChecker
}) => {
  const dateFormat = (0, _kibana.useDateFormat)();
  const timeZone = (0, _kibana.useTimeZone)();
  return (0, _react.useMemo)(() => {
    var _response$;
    const data = response === null || response === void 0 ? void 0 : (_response$ = response[0]) === null || _response$ === void 0 ? void 0 : _response$.data;
    if (!Array.isArray(data)) {
      return [];
    }
    return data.map(action => {
      const isChecked = action.id === selectedValue;
      const timestamp = formatTimestamp(action.startedAt, dateFormat, timeZone);
      const command = action.command;
      const createdBy = action.createdBy;

      // Use the console command name for display (e.g., 'release' instead of 'unisolate')
      const displayCommand = _constants.RESPONSE_ACTION_API_COMMAND_TO_CONSOLE_COMMAND_MAP[command] || command;
      const description = (0, _translations.getPendingActionDescription)(action.id, createdBy, timestamp);

      // Check if user has permission to cancel this action
      const permissionCheck = privilegeChecker ? privilegeChecker(command) : {
        canCancel: false
      };
      const isDisabled = !permissionCheck.canCancel;
      return {
        label: displayCommand,
        value: action.id,
        description,
        data: action,
        checked: isChecked ? 'on' : undefined,
        disabled: isDisabled
      };
    });
  }, [response, selectedValue, privilegeChecker, dateFormat, timeZone]);
};
exports.usePendingActionsOptions = usePendingActionsOptions;