"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.writeTransform = exports.trustedDeviceConditionsToEntries = exports.readTransform = exports.isTrustedDeviceField = exports.entriesToTrustedDeviceConditions = void 0;
var _securitysolutionUtils = require("@kbn/securitysolution-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const OPERATOR_VALUE = 'included';
const createTrustedDeviceEntryMatch = (field, value) => {
  return {
    field,
    value,
    type: 'match',
    operator: OPERATOR_VALUE
  };
};
const createTrustedDeviceEntryWildcard = (field, value) => {
  return {
    field,
    value,
    type: 'wildcard',
    operator: OPERATOR_VALUE
  };
};
const entriesToTrustedDeviceConditions = entries => {
  const result = [];
  entries.forEach(entry => {
    if (!isTrustedDeviceField(entry.field)) {
      return; // Skip non-TD fields
    }
    if (entry.type === 'match' && 'value' in entry) {
      result.push({
        field: entry.field,
        value: entry.value,
        type: entry.type,
        operator: 'included'
      });
    } else if (entry.type === 'wildcard' && 'value' in entry) {
      result.push({
        field: entry.field,
        value: entry.value,
        type: entry.type,
        operator: 'included'
      });
    } else if (entry.type === 'match_any' && 'value' in entry && Array.isArray(entry.value)) {
      result.push({
        field: entry.field,
        value: entry.value,
        type: entry.type,
        operator: 'included'
      });
    }
  });
  return result;
};
exports.entriesToTrustedDeviceConditions = entriesToTrustedDeviceConditions;
const trustedDeviceConditionsToEntries = conditions => {
  const entriesArray = [];
  conditions.forEach(condition => {
    if (!isTrustedDeviceField(condition.field)) {
      return;
    }
    if (condition.type === 'match') {
      entriesArray.push(createTrustedDeviceEntryMatch(condition.field, condition.value));
    } else if (condition.type === 'wildcard') {
      entriesArray.push(createTrustedDeviceEntryWildcard(condition.field, condition.value));
    } else if (condition.type === 'match_any' && Array.isArray(condition.value)) {
      entriesArray.push({
        field: condition.field,
        value: condition.value,
        type: 'match_any',
        operator: OPERATOR_VALUE
      });
    }
  });
  return entriesArray;
};
exports.trustedDeviceConditionsToEntries = trustedDeviceConditionsToEntries;
const isTrustedDeviceField = field => {
  return Object.values(_securitysolutionUtils.TrustedDeviceConditionEntryField).includes(field);
};
exports.isTrustedDeviceField = isTrustedDeviceField;
const readTransform = item => {
  return {
    ...item,
    entries: entriesToTrustedDeviceConditions(item.entries)
  };
};
exports.readTransform = readTransform;
const writeTransform = item => {
  return {
    ...item,
    entries: trustedDeviceConditionsToEntries(item.entries)
  };
};
exports.writeTransform = writeTransform;