"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useMetricAnimation = void 0;
var _react = require("react");
var d3 = _interopRequireWildcard(require("d3"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Custom hook that animates a metric value in the DOM
 * Works by finding the target element and animating its text content
 * Prevents flash by immediately hiding elements and using mutation observer
 */
const useMetricAnimation = ({
  animationDurationMs = 1500,
  selector = '.echMetricText__value'
}) => {
  const animationRef = (0, _react.useRef)(null);
  const hasAnimatedRef = (0, _react.useRef)(false);
  const observerRef = (0, _react.useRef)(null);
  (0, _react.useEffect)(() => {
    // Reset the animation flag for each render
    hasAnimatedRef.current = false;
    let elementFound = false;
    let animationStarted = false;
    let observer = null;
    const startAnimation = element => {
      if (animationStarted) return;
      animationStarted = true;

      // Store the original text content and styling
      const originalText = element.textContent || '';

      // Extract the numeric value from the text (remove currency symbols, commas, etc.)
      const numericValue = parseFloat(originalText.replace(/[^0-9.-]/g, ''));
      if (isNaN(numericValue)) {
        if (observer) {
          observer.disconnect();
        }
        return;
      }

      // Immediately hide the element to prevent flash
      element.style.opacity = '0';

      // Add a small delay to ensure the element is fully styled
      setTimeout(() => {
        // Show the element and start animation
        element.style.opacity = '1';

        // Set initial animated value
        element.textContent = '$0';

        // Animate the counter directly on the original element
        const selection = d3.select(element);
        selection.transition().duration(animationDurationMs).tween('text', function () {
          const interpolator = d3.interpolateNumber(0, numericValue);
          return t => {
            const currentValue = Math.round(interpolator(t));
            // Format the number with currency symbol and commas
            const formattedValue = `$${currentValue.toLocaleString()}`;
            this.textContent = formattedValue;
          };
        });

        // Handle the end of animation using setTimeout instead of transition.on
        setTimeout(() => {
          // Restore original text content
          element.textContent = originalText;

          // Restore opacity
          element.style.opacity = '';
          if (observer) {
            observer.disconnect();
          }
          hasAnimatedRef.current = true;
        }, animationDurationMs);
        animationRef.current = selection;
      }, 100); // 100ms delay to ensure element is fully styled
    };

    // First, check if the element already exists
    const checkExistingElement = () => {
      const existingElement = document.querySelector(selector);
      if (existingElement && !elementFound && !animationStarted) {
        elementFound = true;
        startAnimation(existingElement);
        return true;
      }
      return false;
    };

    // Check immediately
    if (checkExistingElement()) {
      return;
    }

    // Set up a mutation observer to catch new elements as they're added
    observer = new MutationObserver(mutations => {
      if (animationStarted) return;
      mutations.forEach(mutation => {
        mutation.addedNodes.forEach(node => {
          if (node.nodeType === Node.ELEMENT_NODE) {
            const element = node;
            const targetElement = element.querySelector(selector);
            if (targetElement && !elementFound) {
              elementFound = true;
              startAnimation(targetElement);
            }
            // Also check if the node itself is the target
            if (element.matches && element.matches(selector) && !elementFound) {
              elementFound = true;
              startAnimation(element);
            }
          }
        });
      });
    });

    // Start observing
    observer.observe(document.body, {
      childList: true,
      subtree: true
    });
    observerRef.current = observer;

    // Fallback: check for existing element after a short delay
    const fallbackTimeout = setTimeout(() => {
      if (!elementFound && !animationStarted) {
        const element = document.querySelector(selector);
        if (element) {
          elementFound = true;
          startAnimation(element);
        }
      }
    }, 50); // Reduced from 200ms to 50ms for faster detection

    return () => {
      clearTimeout(fallbackTimeout);
      if (observerRef.current) {
        observerRef.current.disconnect();
      }
      if (animationRef.current) {
        animationRef.current.interrupt();
      }
    };
  }, [animationDurationMs, selector]);
};
exports.useMetricAnimation = useMetricAnimation;