"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SiemRulesMigrationsService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _constants = require("../../../../common/siem_migrations/constants");
var api = _interopRequireWildcard(require("../api"));
var i18n = _interopRequireWildcard(require("./translations"));
var _telemetry = require("./telemetry");
var _service = require("../../common/service");
var _success_notification = require("./notification/success_notification");
var _constants2 = require("../../common/constants");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CREATE_MIGRATION_BODY_BATCH_SIZE = 50;
class SiemRulesMigrationsService extends _service.SiemMigrationsServiceBase {
  constructor(core, plugins, telemetryService) {
    super(core, plugins);
    (0, _defineProperty2.default)(this, "telemetry", void 0);
    this.telemetry = new _telemetry.SiemRulesMigrationsTelemetry(telemetryService);
  }

  /** Accessor for the rule migrations API client */
  get api() {
    return api;
  }

  /** Adds rules to a rule migration, batching the requests to avoid hitting the max payload size limit of the API */
  async addRulesToMigration(migrationId, rules) {
    const rulesCount = rules.length;
    if (rulesCount === 0) {
      throw new Error(i18n.EMPTY_RULES_ERROR);
    }

    // Batching creation to avoid hitting the max payload size limit of the API
    for (let i = 0; i < rulesCount; i += CREATE_MIGRATION_BODY_BATCH_SIZE) {
      const rulesBatch = rules.slice(i, i + CREATE_MIGRATION_BODY_BATCH_SIZE);
      await api.addRulesToMigration({
        migrationId,
        body: rulesBatch
      });
    }
  }

  /** Creates a rule migration with a name and adds the rules to it, returning the migration ID */
  async createRuleMigration(data, migrationName) {
    const rulesCount = data.length;
    if (rulesCount === 0) {
      throw new Error(i18n.EMPTY_RULES_ERROR);
    }
    try {
      // create the migration
      const {
        migration_id: migrationId
      } = await api.createRuleMigration({
        name: migrationName
      });
      await this.addRulesToMigration(migrationId, data);
      this.telemetry.reportSetupMigrationCreated({
        migrationId,
        rulesCount
      });
      return migrationId;
    } catch (error) {
      this.telemetry.reportSetupMigrationCreated({
        rulesCount,
        error
      });
      throw error;
    }
  }

  /** Deletes a rule migration by its ID, refreshing the stats to remove it from the list */
  async deleteMigration(migrationId) {
    try {
      await api.deleteMigration({
        migrationId
      });

      // Refresh stats to remove the deleted migration from the list. All UI observables will be updated automatically
      await this.getMigrationsStats();
      this.telemetry.reportSetupMigrationDeleted({
        migrationId
      });
      return migrationId;
    } catch (error) {
      this.telemetry.reportSetupMigrationDeleted({
        migrationId,
        error
      });
      throw error;
    }
  }

  /** Upserts resources for a rule migration, batching the requests to avoid hitting the max payload size limit of the API */
  async upsertMigrationResources(migrationId, body) {
    const count = body.length;
    if (count === 0) {
      throw new Error(i18n.EMPTY_RULES_ERROR);
    }
    // We assume all resources are of the same type. There is no use case for mixing types in a single upload
    const type = body[0].type;
    try {
      // Batching creation to avoid hitting the max payload size limit of the API
      for (let i = 0; i < count; i += CREATE_MIGRATION_BODY_BATCH_SIZE) {
        const bodyBatch = body.slice(i, i + CREATE_MIGRATION_BODY_BATCH_SIZE);
        await api.upsertMigrationResources({
          migrationId,
          body: bodyBatch
        });
      }
      this.telemetry.reportSetupResourceUploaded({
        migrationId,
        type,
        count
      });
    } catch (error) {
      this.telemetry.reportSetupResourceUploaded({
        migrationId,
        type,
        count,
        error
      });
      throw error;
    }
  }

  /** Starts a rule migration task and waits for the task to start running */
  async startRuleMigration(migrationId, retry, settings) {
    var _settings$connectorId;
    const missingCapabilities = this.getMissingCapabilities('all');
    if (missingCapabilities.length > 0) {
      this.core.notifications.toasts.add((0, _service.getMissingCapabilitiesToast)(missingCapabilities, this.core));
      return {
        started: false
      };
    }
    const connectorId = (_settings$connectorId = settings === null || settings === void 0 ? void 0 : settings.connectorId) !== null && _settings$connectorId !== void 0 ? _settings$connectorId : this.connectorIdStorage.get();
    const skipPrebuiltRulesMatching = settings === null || settings === void 0 ? void 0 : settings.skipPrebuiltRulesMatching;
    if (!connectorId) {
      this.core.notifications.toasts.add((0, _service.getNoConnectorToast)(this.core));
      return {
        started: false
      };
    }
    const params = {
      migrationId,
      settings: {
        connectorId,
        skipPrebuiltRulesMatching
      },
      retry
    };
    const traceOptions = this.traceOptionsStorage.get();
    if (traceOptions) {
      params.langSmithOptions = {
        project_name: traceOptions.langSmithProject,
        api_key: traceOptions.langSmithApiKey
      };
    }
    try {
      const result = await api.startRuleMigration(params);

      // Should take a few seconds to start the task, so we poll until it is running
      await this.migrationTaskPollingUntil(migrationId, ({
        status
      }) => status === _constants.SiemMigrationTaskStatus.RUNNING, {
        sleepSecs: _constants2.START_STOP_POLLING_SLEEP_SECONDS,
        timeoutSecs: 90
      } // wait up to 90 seconds for the task to start
      );
      this.startPolling();
      this.telemetry.reportStartTranslation(params);
      return result;
    } catch (error) {
      this.telemetry.reportStartTranslation({
        ...params,
        error
      });
      throw error;
    }
  }

  /** Stops a running rule migration task and waits for the task to completely stop */
  async stopRuleMigration(migrationId) {
    const missingCapabilities = this.getMissingCapabilities('all');
    if (missingCapabilities.length > 0) {
      this.core.notifications.toasts.add((0, _service.getMissingCapabilitiesToast)(missingCapabilities, this.core));
      return {
        stopped: false
      };
    }
    const params = {
      migrationId
    };
    try {
      const result = await api.stopRuleMigration(params);

      // Should take a few seconds to stop the task, so we poll until it is not running anymore
      await this.migrationTaskPollingUntil(migrationId, ({
        status
      }) => status !== _constants.SiemMigrationTaskStatus.RUNNING,
      // may be STOPPED, FINISHED or INTERRUPTED
      {
        sleepSecs: _constants2.START_STOP_POLLING_SLEEP_SECONDS,
        timeoutSecs: 90
      } // wait up to 90 seconds for the task to stop
      );
      this.telemetry.reportStopTranslation(params);
      return result;
    } catch (error) {
      this.telemetry.reportStopTranslation({
        ...params,
        error
      });
      throw error;
    }
  }
  async startMigrationFromStats(connectorId, taskStats) {
    var _taskStats$last_execu;
    const skipPrebuiltRulesMatching = (_taskStats$last_execu = taskStats.last_execution) === null || _taskStats$last_execu === void 0 ? void 0 : _taskStats$last_execu.skip_prebuilt_rules_matching;
    await api.startRuleMigration({
      migrationId: taskStats.id,
      settings: {
        connectorId,
        skipPrebuiltRulesMatching
      }
    });
  }
  async fetchMigrationStats({
    migrationId
  }) {
    const stats = await api.getRuleMigrationStats({
      migrationId
    });
    return stats;
  }
  async fetchMigrationsStatsAll(params) {
    const allStats = await api.getRuleMigrationsStatsAll(params);
    return allStats;
  }
  sendFinishedMigrationNotification(taskStats) {
    (0, _success_notification.raiseSuccessToast)(taskStats, this.core);
  }
}
exports.SiemRulesMigrationsService = SiemRulesMigrationsService;