"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchSourceDocuments = void 0;
var _perf_hooks = require("perf_hooks");
var _logged_requests = require("../utils/logged_requests");
var i18n = _interopRequireWildcard(require("../translations"));
var _build_esql_search_request = require("./build_esql_search_request");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * fetches source documents by list of their ids
 * it used for a case when non-aggregating has _id property to enrich alert with source document,
 * if some of the properties missed from resulted query
 */
const fetchSourceDocuments = async ({
  isRuleAggregating,
  results,
  esClient,
  index,
  loggedRequests,
  hasLoggedRequestsReachedLimit,
  runtimeMappings,
  excludedDocuments
}) => {
  const ids = results.reduce((acc, doc) => {
    if (doc._id) {
      acc.push(doc._id);
    }
    return acc;
  }, []);

  // we will fetch source documents only for non-aggregating rules, since aggregating do not have _id
  if (ids.length === 0 || isRuleAggregating) {
    return {};
  }
  const idsQuery = {
    query: {
      bool: {
        filter: {
          ids: {
            values: ids
          }
        },
        ...(Object.keys(excludedDocuments).length > 0 ? {
          must_not: (0, _build_esql_search_request.convertExternalIdsToDSL)(excludedDocuments, new Set(ids))
        } : {})
      }
    }
  };
  const searchBody = {
    query: idsQuery.query,
    _source: true,
    fields: ['*'],
    size: 2 * ids.length,
    // allows supporting multiple documents with the same _id across different indices
    runtime_mappings: runtimeMappings
  };
  const ignoreUnavailable = true;
  if (loggedRequests) {
    loggedRequests.push({
      request: hasLoggedRequestsReachedLimit ? undefined : (0, _logged_requests.logQueryRequest)(searchBody, {
        index,
        ignoreUnavailable
      }),
      description: i18n.FIND_SOURCE_DOCUMENTS_REQUEST_DESCRIPTION
    });
  }
  const searchStart = _perf_hooks.performance.now();
  const response = await esClient.search({
    index,
    ...searchBody,
    ignore_unavailable: ignoreUnavailable
  });
  if (loggedRequests) {
    loggedRequests[loggedRequests.length - 1].duration = Math.round(_perf_hooks.performance.now() - searchStart);
  }
  return response.hits.hits.reduce((acc, hit) => {
    if (hit._id) {
      if (!acc[hit._id]) {
        acc[hit._id] = [];
      }
      acc[hit._id].push({
        fields: hit.fields,
        _source: hit._source,
        _id: hit._id,
        _index: hit._index,
        _version: hit._version
      });
    }
    return acc;
  }, {});
};
exports.fetchSourceDocuments = fetchSourceDocuments;