"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformBulkCreatedItemsToHits = exports.shouldFilterByCardinality = exports.searchResultHasAggs = exports.getThresholdTermsHash = exports.getSignalHistory = exports.calculateThresholdSignalUuid = void 0;
var _crypto = require("crypto");
var _uuid = require("uuid");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Returns a new signal history based on what the previous
 * threshold rule state had stored and what the current rule
 * run tuple timestamp is.
 *
 * This is used to determine which terms buckets over
 * which periods of time are to be used in the search after
 *
 * @param state ThresholdAlertState
 * @param signalHistory ThresholdSignalHistory
 * @param tuple RuleRangeTuple
 * @returns ThresholdSignalHistory
 */
const getSignalHistory = (state, signalHistory, tuple) => {
  if (state.initialized) {
    return Object.entries(signalHistory).reduce((acc, [hash, entry]) => {
      if (entry.lastSignalTimestamp > tuple.from.valueOf()) {
        acc[hash] = entry;
        return acc;
      } else {
        return acc;
      }
    }, {});
  }
  return signalHistory;
};
exports.getSignalHistory = getSignalHistory;
const shouldFilterByCardinality = threshold => {
  var _threshold$cardinalit;
  return !!((_threshold$cardinalit = threshold.cardinality) !== null && _threshold$cardinalit !== void 0 && _threshold$cardinalit.length);
};
exports.shouldFilterByCardinality = shouldFilterByCardinality;
const calculateThresholdSignalUuid = (ruleId, startedAt, thresholdFields, key) => {
  // used to generate stable Threshold Signals ID when run with the same params
  const NAMESPACE_ID = '0684ec03-7201-4ee0-8ee0-3a3f6b2479b2';
  const startedAtString = startedAt.toISOString();
  const keyString = key !== null && key !== void 0 ? key : '';
  const baseString = `${ruleId}${startedAtString}${thresholdFields.join(',')}${keyString}`;
  return (0, _uuid.v5)(baseString, NAMESPACE_ID);
};
exports.calculateThresholdSignalUuid = calculateThresholdSignalUuid;
const getThresholdTermsHash = terms => {
  return (0, _crypto.createHash)('sha256').update(terms.sort((term1, term2) => term1.field > term2.field ? 1 : -1).map(term => {
    return `${term.field}:${term.value}`;
  }).join(',')).digest('hex');
};
exports.getThresholdTermsHash = getThresholdTermsHash;
const searchResultHasAggs = obj => (obj === null || obj === void 0 ? void 0 : obj.aggregations) != null;

/**
 * transforms documents returned from bulk creation into Hit formatting
 * basically, moving all fields(apart from _id & _index) from root node to _source property
 * { _id: 1, _index: "logs", field1, field2 } => { _id: 1, _index: "logs", _source: { field1, field2 } }
 */
exports.searchResultHasAggs = searchResultHasAggs;
const transformBulkCreatedItemsToHits = items => items.map(alert => {
  const {
    _id,
    _index,
    ...source
  } = alert;
  return {
    _id,
    _index,
    _source: {
      ...source
    }
  };
});
exports.transformBulkCreatedItemsToHits = transformBulkCreatedItemsToHits;