"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getESQL = exports.getCompositeQuery = exports.calculateScoresWithESQL = exports.buildRiskScoreBucket = void 0;
var _lodash = require("lodash");
var _technical_rule_data_field_names = require("@kbn/rule-registry-plugin/common/technical_rule_data_field_names");
var _Record = require("fp-ts/Record");
var _types = require("../../../../common/entity_analytics/types");
var _utils = require("../../../../common/entity_analytics/utils");
var _with_security_span = require("../../../utils/with_security_span");
var _constants = require("./constants");
var _calculate_risk_scores = require("./calculate_risk_scores");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const calculateScoresWithESQL = async params => (0, _with_security_span.withSecuritySpan)('calculateRiskScores', async () => {
  const {
    afterKeys,
    alertSampleSizePerShard,
    assetCriticalityService,
    esClient,
    identifierType,
    index,
    logger,
    pageSize,
    weights
  } = params;
  const now = new Date().toISOString();
  const filter = getFilters(params);
  const identifierTypes = identifierType ? [identifierType] : (0, _utils.getEntityAnalyticsEntityTypes)();
  const compositeQuery = getCompositeQuery(identifierTypes, filter, params);
  logger.trace(`STEP ONE: Executing ESQL Risk Score composite query:\n${JSON.stringify(compositeQuery)}`);
  const response = await esClient.search(compositeQuery).catch(e => {
    logger.error(`Error executing composite query: ${e.message}`);
  });
  if (!(response !== null && response !== void 0 && response.aggregations)) {
    return {
      after_keys: {},
      scores: {
        host: [],
        user: [],
        service: []
      },
      entities: {
        user: [],
        service: [],
        host: [],
        generic: []
      }
    };
  }
  const promises = (0, _Record.toEntries)(response.aggregations).map(([entityType, {
    buckets,
    after_key: afterKey
  }]) => {
    var _afterKeys$entityType;
    const entities = buckets.map(({
      key
    }) => key[_types.EntityTypeToIdentifierField[entityType]]);
    if (entities.length === 0) {
      return Promise.resolve([entityType, {
        afterKey: afterKey,
        scores: []
      }, entities]);
    }
    const bounds = {
      lower: (_afterKeys$entityType = afterKeys[entityType]) === null || _afterKeys$entityType === void 0 ? void 0 : _afterKeys$entityType[_types.EntityTypeToIdentifierField[entityType]],
      upper: afterKey === null || afterKey === void 0 ? void 0 : afterKey[_types.EntityTypeToIdentifierField[entityType]]
    };
    const weight = (0, _calculate_risk_scores.getGlobalWeightForIdentifierType)(entityType, weights) || 1;
    const query = getESQL({
      entityType: entityType,
      bounds,
      sampleSize: alertSampleSizePerShard || 10000,
      pageSize,
      index,
      weight
    });
    return esClient.esql.query({
      query,
      filter: {
        bool: {
          filter
        }
      }
    }).then(rs => rs.values.map(buildRiskScoreBucket(entityType, index, weight))).then(riskScoreBuckets => {
      return (0, _calculate_risk_scores.processScores)({
        assetCriticalityService,
        buckets: riskScoreBuckets,
        identifierField: _types.EntityTypeToIdentifierField[entityType],
        logger,
        now
      });
    }).then(scores => {
      return [entityType, {
        scores,
        afterKey: afterKey
      }, entities];
    }).catch(error => {
      logger.error(`Error executing ESQL query for entity type ${entityType}: ${error.message}`);
      logger.error(`Query: ${query}`);
      return [entityType, {
        afterKey: afterKey,
        scores: []
      }, entities];
    });
  });
  const esqlResults = await Promise.all(promises);
  const results = esqlResults.reduce((res, [entityType, {
    afterKey,
    scores
  }, entities]) => {
    res.after_keys[entityType] = afterKey;
    res.scores[entityType] = scores;
    res.entities[entityType] = entities;
    return res;
  }, {
    after_keys: {},
    scores: {},
    entities: {
      user: [],
      service: [],
      host: [],
      generic: []
    }
  });
  return results;
});
exports.calculateScoresWithESQL = calculateScoresWithESQL;
const getFilters = options => {
  const {
    excludeAlertStatuses = [],
    excludeAlertTags = [],
    range,
    filter: userFilter
  } = options;
  const filters = [(0, _calculate_risk_scores.filterFromRange)(range), {
    exists: {
      field: _technical_rule_data_field_names.ALERT_RISK_SCORE
    }
  }];
  if (excludeAlertStatuses.length > 0) {
    filters.push({
      bool: {
        must_not: {
          terms: {
            [_technical_rule_data_field_names.ALERT_WORKFLOW_STATUS]: excludeAlertStatuses
          }
        }
      }
    });
  }
  if (!(0, _lodash.isEmpty)(userFilter)) {
    filters.push(userFilter);
  }
  if (excludeAlertTags.length > 0) {
    filters.push({
      bool: {
        must_not: {
          terms: {
            [_technical_rule_data_field_names.ALERT_WORKFLOW_TAGS]: excludeAlertTags
          }
        }
      }
    });
  }
  return filters;
};
const getCompositeQuery = (entityTypes, filter, params) => {
  const {
    index,
    pageSize,
    runtimeMappings,
    afterKeys
  } = params;
  return {
    size: 0,
    index,
    ignore_unavailable: true,
    runtime_mappings: runtimeMappings,
    query: {
      function_score: {
        query: {
          bool: {
            filter,
            should: [{
              match_all: {} // This forces ES to calculate score
            }]
          }
        },
        field_value_factor: {
          field: _technical_rule_data_field_names.ALERT_RISK_SCORE // sort by risk score
        }
      }
    },
    aggs: entityTypes.reduce((aggs, entityType) => {
      const idField = _types.EntityTypeToIdentifierField[entityType];
      return {
        ...aggs,
        [entityType]: {
          composite: {
            size: pageSize,
            sources: [{
              [idField]: {
                terms: {
                  field: idField
                }
              }
            }],
            after: afterKeys[entityType]
          }
        }
      };
    }, {})
  };
};
exports.getCompositeQuery = getCompositeQuery;
const getESQL = ({
  entityType,
  bounds,
  sampleSize,
  pageSize,
  index,
  weight
}) => {
  const identifierField = _types.EntityTypeToIdentifierField[entityType];
  const lower = bounds.lower ? `${identifierField} > ${bounds.lower}` : undefined;
  const upper = bounds.upper ? `${identifierField} <= ${bounds.upper}` : undefined;
  if (!lower && !upper) {
    throw new Error('Either lower or upper after key must be provided for pagination');
  }
  const rangeClause = [lower, upper].filter(Boolean).join(' and ');
  const query = /* ESQL */`
  FROM ${index} METADATA _index
    | WHERE kibana.alert.risk_score IS NOT NULL AND KQL("${rangeClause}")
    | RENAME kibana.alert.risk_score as risk_score,
             kibana.alert.rule.name as rule_name,
             kibana.alert.rule.uuid as rule_id,
             kibana.alert.uuid as alert_id,
             event.kind as category,
             @timestamp as time
    | EVAL input = CONCAT(""" {"score": """", risk_score::keyword, """", "time": """", time::keyword, """", "index": """", _index, """", "rule_name": """", rule_name, """\", "category": """", category, """\", "id": \"""", alert_id, """\" } """)
    | STATS
        alert_count = count(risk_score),
        scores = ${weight} * MV_PSERIES_WEIGHTED_SUM(TOP(risk_score, ${sampleSize}, "desc"), ${_constants.RIEMANN_ZETA_S_VALUE}),
        risk_inputs = TOP(input, 10, "desc")
    BY ${identifierField}
    | SORT scores DESC, ${identifierField} ASC
    | LIMIT ${pageSize}
  `;
  return query;
};
exports.getESQL = getESQL;
const buildRiskScoreBucket = (entityType, index, weight = 1) => row => {
  const [count, score, _inputs, entity] = row;
  const inputs = (Array.isArray(_inputs) ? _inputs : [_inputs]).map((input, i) => {
    const parsedRiskInputData = JSON.parse(input);
    const value = parseFloat(parsedRiskInputData.score);
    const currentScore = value / Math.pow(i + 1, _constants.RIEMANN_ZETA_S_VALUE);
    return {
      ...parsedRiskInputData,
      score: value,
      contribution: currentScore / _constants.RIEMANN_ZETA_VALUE,
      index
    };
  });
  return {
    key: {
      [_types.EntityTypeToIdentifierField[entityType]]: entity
    },
    doc_count: count,
    top_inputs: {
      doc_count: inputs.length,
      risk_details: {
        value: {
          score,
          normalized_score: score / _constants.RIEMANN_ZETA_VALUE,
          // normalize value to be between 0-100
          notes: [],
          category_1_score: score / weight,
          // category score before global weight applied and normalization
          category_1_count: count,
          risk_inputs: inputs
        }
      }
    }
  };
};
exports.buildRiskScoreBucket = buildRiskScoreBucket;