"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SiemMigrationTaskRunner = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _assert = _interopRequireDefault(require("assert"));
var _server = require("@kbn/kibana-utils-plugin/server");
var _constants = require("../../../../../common/siem_migrations/constants");
var _promise_pool = require("../../../../utils/promise_pool");
var _comments = require("./util/comments");
var _actions_client_chat = require("./util/actions_client_chat");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/** Number of items loaded in memory to be translated in the pool */
const TASK_BATCH_SIZE = 100;
/** The timeout of each individual agent invocation in minutes */
const AGENT_INVOKE_TIMEOUT_MIN = 20;

/** Exponential backoff configuration to handle rate limit errors */
const RETRY_CONFIG = {
  initialRetryDelaySeconds: 1,
  backoffMultiplier: 2,
  maxRetries: 8
  // max waiting time 4m15s (1*2^8 = 256s)
};

/** Executor sleep configuration
 * A sleep time applied at the beginning of each single item translation in the execution pool,
 * The objective of this sleep is to spread the load of concurrent translations, and prevent hitting the rate limit repeatedly.
 * The sleep time applied is a random number between [0-value]. Every time we hit rate limit the value is increased by the multiplier, up to the limit.
 */
const EXECUTOR_SLEEP = {
  initialValueSeconds: 3,
  multiplier: 2,
  limitSeconds: 96 // 1m36s (5 increases)
};

/** This limit should never be reached, it's a safety net to prevent infinite loops.
 * It represents the max number of consecutive rate limit recovery & failure attempts.
 * This can only happen when the API can not process all concurrenct translations ( based on taskConcurrency ) at a time,
 * even after the executor sleep is increased on every attempt.
 **/
const EXECUTOR_RECOVER_MAX_ATTEMPTS = 3;
class SiemMigrationTaskRunner {
  /** Number of concurrent items to process. Each item triggers one instance of graph */

  constructor(migrationId, request, startedBy, abortController, data, logger, dependencies) {
    (0, _defineProperty2.default)(this, "telemetry", void 0);
    (0, _defineProperty2.default)(this, "task", void 0);
    (0, _defineProperty2.default)(this, "abort", void 0);
    (0, _defineProperty2.default)(this, "executorSleepMultiplier", EXECUTOR_SLEEP.initialValueSeconds);
    (0, _defineProperty2.default)(this, "isWaiting", false);
    /** Creates the task invoke function, the input is prepared and the output is processed as a migrationItem */
    (0, _defineProperty2.default)(this, "createTaskInvoke", async (migrationItem, config) => {
      const input = await this.prepareTaskInput(migrationItem);
      return async () => {
        const output = await this.executeTask(input, config);
        return this.processTaskOutput(migrationItem, output);
      };
    });
    (0, _defineProperty2.default)(this, "executorSleep", async () => {
      const seconds = Math.random() * this.executorSleepMultiplier;
      this.logger.debug(`Executor sleep: ${seconds.toFixed(3)}s`);
      await this.sleep(seconds);
    });
    (0, _defineProperty2.default)(this, "increaseExecutorSleep", () => {
      const increasedMultiplier = this.executorSleepMultiplier * EXECUTOR_SLEEP.multiplier;
      if (increasedMultiplier > EXECUTOR_SLEEP.limitSeconds) {
        this.logger.warn('Executor sleep reached the maximum value');
        return;
      }
      this.executorSleepMultiplier = increasedMultiplier;
    });
    this.migrationId = migrationId;
    this.request = request;
    this.startedBy = startedBy;
    this.abortController = abortController;
    this.data = data;
    this.logger = logger;
    this.dependencies = dependencies;
    this.actionsClientChat = new _actions_client_chat.ActionsClientChat(this.request, this.dependencies);
    this.abort = (0, _server.abortSignalToPromise)(this.abortController.signal);
  }

  /** Receives the connectorId and creates the `this.task` and `this.telemetry` attributes */

  /** Prepares the migration item for the task execution */

  /** Processes the output of the migration task and returns the item to save */

  /** Optional initialization logic */
  async initialize() {}
  async run(invocationConfig) {
    (0, _assert.default)(this.telemetry, 'telemetry is missing please call setup() first');
    const {
      telemetry,
      migrationId
    } = this;
    const migrationTaskTelemetry = telemetry.startSiemMigrationTask();
    try {
      this.logger.debug('Initializing migration');
      await this.withAbort(this.initialize());
    } catch (error) {
      migrationTaskTelemetry.failure(error);
      if (error instanceof _server.AbortError) {
        this.logger.info('Abort signal received, stopping initialization');
        return;
      } else {
        throw new Error(`Migration initialization failed. ${error}`);
      }
    }
    const migrateItemTask = this.createMigrateItemTask(invocationConfig);
    this.logger.debug(`Started translations. Concurrency is: ${this.taskConcurrency}`);
    try {
      do {
        const {
          data: migrationItems
        } = await this.data.items.get(migrationId, {
          filters: {
            status: _constants.SiemMigrationStatus.PENDING
          },
          size: TASK_BATCH_SIZE // keep these items in memory and process them in the promise pool with concurrency limit
        });
        if (migrationItems.length === 0) {
          break;
        }
        this.logger.debug(`Start processing batch of ${migrationItems.length} items`);
        const {
          errors
        } = await (0, _promise_pool.initPromisePool)({
          concurrency: this.taskConcurrency,
          abortSignal: this.abortController.signal,
          items: migrationItems,
          executor: async migrationItem => {
            const itemTranslationTelemetry = migrationTaskTelemetry.startItemTranslation();
            try {
              await this.saveItemProcessing(migrationItem);
              const migratedItem = await migrateItemTask(migrationItem);
              await this.saveItemCompleted(migratedItem);
              itemTranslationTelemetry.success(migratedItem);
            } catch (error) {
              if (this.abortController.signal.aborted) {
                throw new _server.AbortError();
              }
              itemTranslationTelemetry.failure(error);
              await this.saveItemFailed(migrationItem, error);
            }
          }
        });
        if (errors.length > 0) {
          throw errors[0].error; // Only AbortError is thrown from the pool. The task was aborted
        }
        this.logger.debug('Batch processed successfully');
      } while (true);
      migrationTaskTelemetry.success();
      this.logger.info('Migration completed successfully');
    } catch (error) {
      await this.data.items.releaseProcessing(migrationId);
      if (error instanceof _server.AbortError) {
        migrationTaskTelemetry.aborted(error);
        this.logger.info('Abort signal received, stopping migration');
      } else {
        migrationTaskTelemetry.failure(error);
        throw new Error(`Error processing migration: ${error}`);
      }
    } finally {
      this.abort.cleanup();
    }
  }

  /** Executes the task with raw input and config, and returns the output promise. */
  async executeTask(input, config) {
    (0, _assert.default)(this.task, 'Migration task is not defined');
    return this.task(input, config);
  }
  createMigrateItemTask(invocationConfig) {
    const config = {
      timeout: AGENT_INVOKE_TIMEOUT_MIN * 60 * 1000,
      // milliseconds timeout
      ...invocationConfig,
      metadata: {
        migrationId: this.migrationId
      },
      signal: this.abortController.signal
    };

    // Invokes the item translation with exponential backoff, should be called only when the rate limit has been hit
    const invokeWithBackoff = async invoke => {
      this.logger.debug('Rate limit backoff started');
      let retriesLeft = RETRY_CONFIG.maxRetries;
      while (true) {
        try {
          await this.sleepRetry(retriesLeft);
          retriesLeft--;
          const result = await invoke();
          this.logger.info(`Rate limit backoff completed successfully after ${RETRY_CONFIG.maxRetries - retriesLeft} retries`);
          return result;
        } catch (error) {
          if (!this.isRateLimitError(error) || retriesLeft === 0) {
            const logMessage = retriesLeft === 0 ? `Rate limit backoff completed unsuccessfully` : `Rate limit backoff interrupted. ${error} `;
            this.logger.debug(logMessage);
            throw error;
          }
          this.logger.debug(`Rate limit backoff not completed, retries left: ${retriesLeft}`);
        }
      }
    };
    let backoffPromise;
    // Migrates one item, this function will be called concurrently by the promise pool.
    // Handles rate limit errors and ensures only one task is executing the backoff retries at a time, the rest of translation will await.
    const migrateItem = async migrationItem => {
      const invoke = await this.createTaskInvoke(migrationItem, config);
      let recoverAttemptsLeft = EXECUTOR_RECOVER_MAX_ATTEMPTS;
      while (true) {
        try {
          await this.executorSleep(); // Random sleep, increased every time we hit the rate limit.
          return await invoke();
        } catch (error) {
          this.logger.debug(`Error during migration item translation: ${error.toString()}`);
          if (!this.isRateLimitError(error) || recoverAttemptsLeft === 0) {
            throw error;
          }
          if (!backoffPromise) {
            // only one translation handles the rate limit backoff retries, the rest will await it and try again when it's resolved
            backoffPromise = invokeWithBackoff(invoke);
            this.isWaiting = true;
            return backoffPromise.finally(() => {
              backoffPromise = undefined;
              this.increaseExecutorSleep();
              this.isWaiting = false;
            });
          }
          this.logger.debug(`Awaiting backoff task for migration item "${migrationItem.id}"`);
          await backoffPromise.catch(() => {
            throw error; // throw the original error
          });
          recoverAttemptsLeft--;
        }
      }
    };
    return migrateItem;
  }
  isRateLimitError(error) {
    return error.message.match(/\b429\b/); // "429" (whole word in the error message): Too Many Requests.
  }
  async withAbort(promise) {
    return Promise.race([promise, this.abort.promise]);
  }
  async sleep(seconds) {
    await this.withAbort(new Promise(resolve => setTimeout(resolve, seconds * 1000)));
  }

  // Exponential backoff implementation
  async sleepRetry(retriesLeft) {
    const seconds = RETRY_CONFIG.initialRetryDelaySeconds * Math.pow(RETRY_CONFIG.backoffMultiplier, RETRY_CONFIG.maxRetries - retriesLeft);
    this.logger.debug(`Retry sleep: ${seconds}s`);
    await this.sleep(seconds);
  }
  async saveItemProcessing(migrationItem) {
    this.logger.debug(`Starting translation of document "${migrationItem.id}"`);
    return this.data.items.saveProcessing(migrationItem.id);
  }
  async saveItemCompleted(migrationItem) {
    this.logger.debug(`Translation of document "${migrationItem.id}" succeeded`);
    return this.data.items.saveCompleted(migrationItem);
  }
  async saveItemFailed(migrationItem, error) {
    this.logger.error(`Error translating migration item "${migrationItem.id}" with error: ${error.message}`);
    const comments = [(0, _comments.generateAssistantComment)(`Error migrating document: ${error.message}`)];
    return this.data.items.saveError({
      ...migrationItem,
      comments
    });
  }
}
exports.SiemMigrationTaskRunner = SiemMigrationTaskRunner;