"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ElasticsearchCircuitBreaker = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Configuration interface for Elasticsearch Circuit Breaker.
 */

class ElasticsearchCircuitBreaker extends _utils.BaseCircuitBreaker {
  constructor(config, client) {
    super();
    (0, _defineProperty2.default)(this, "lastHealth", 'green');
    (0, _defineProperty2.default)(this, "lastJvmStats", void 0);
    (0, _defineProperty2.default)(this, "lastCpuStats", void 0);
    (0, _defineProperty2.default)(this, "nodeTimestamps", {});
    this.config = config;
    this.client = client;
    if (config.maxJvmHeapUsedPercent < 0 || config.maxJvmHeapUsedPercent > 100) {
      throw new Error('maxJvmHeapUsedPercent must be between 0 and 100');
    }
  }
  async validate() {
    try {
      const healthResp = await this.client.cluster.health();
      this.lastHealth = healthResp.status;
      const status = this.lastHealth;
      if (!this.config.expectedClusterHealth.includes(status)) {
        return this.failure(`Elasticsearch cluster health is ${status}`);
      }
      const nodesResp = await this.client.nodes.stats({
        metric: ['jvm', 'os']
      });
      if (!nodesResp.nodes || Object.keys(nodesResp.nodes).length === 0) {
        return this.failure('No Elasticsearch nodes found');
      }
      this.lastJvmStats = {};
      this.lastCpuStats = {};
      for (const nodeId of Object.keys(nodesResp.nodes)) {
        var _jvm$mem, _os$cpu;
        const node = nodesResp.nodes[nodeId];
        const currentTimestamp = node.timestamp;
        const lastReportedTimestamp = this.nodeTimestamps[nodeId];
        if (currentTimestamp === undefined || lastReportedTimestamp === currentTimestamp) {
          return this.failure(`Node ${nodeId} is stale: no timestamp updates detected. Current timestamp=${currentTimestamp}, Last reported timestamp=${lastReportedTimestamp}`);
        }
        this.nodeTimestamps[nodeId] = currentTimestamp;
        const jvm = node.jvm;
        const os = node.os;
        if ((jvm === null || jvm === void 0 ? void 0 : (_jvm$mem = jvm.mem) === null || _jvm$mem === void 0 ? void 0 : _jvm$mem.heap_used_percent) !== undefined && (os === null || os === void 0 ? void 0 : (_os$cpu = os.cpu) === null || _os$cpu === void 0 ? void 0 : _os$cpu.percent) !== undefined) {
          const heapUsedPercent = jvm.mem.heap_used_percent;
          const cpuPercent = os.cpu.percent;
          this.lastJvmStats[nodeId] = heapUsedPercent;
          this.lastCpuStats[nodeId] = cpuPercent;
          if (heapUsedPercent > this.config.maxJvmHeapUsedPercent) {
            return this.failure(`Node ${nodeId} JVM heap used ${heapUsedPercent}% exceeds threshold`);
          }
          if (cpuPercent > this.config.maxCpuPercent) {
            return this.failure(`Node ${nodeId} CPU usage ${cpuPercent}% exceeds threshold`);
          }
        } else {
          return this.failure(`Node ${nodeId} missing metrics. JVM: ${jvm === null || jvm === void 0 ? void 0 : jvm.mem}, OS: ${os === null || os === void 0 ? void 0 : os.cpu}`);
        }
      }
      return this.success();
    } catch (error) {
      return this.failure(`Failed to get ES cluster or node stats: ${error.message}`);
    }
  }
  stats() {
    return {
      clusterHealth: this.lastHealth,
      jvmHeapUsedPercentPerNode: this.lastJvmStats,
      cpuPercentPerNode: this.lastCpuStats
    };
  }
  validationIntervalMs() {
    return this.config.validationIntervalMs;
  }
}
exports.ElasticsearchCircuitBreaker = ElasticsearchCircuitBreaker;