"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GetHistogramIndicatorAggregation = void 0;
var _common = require("../transform_generators/common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class GetHistogramIndicatorAggregation {
  constructor(indicator) {
    this.indicator = indicator;
  }
  buildAggregation(indicator) {
    const filter = indicator.filter ? (0, _common.getElasticsearchQueryOrThrow)(indicator.filter) : {
      match_all: {}
    };
    if (indicator.aggregation === 'value_count') {
      return {
        filter,
        aggs: {
          total: {
            value_count: {
              field: indicator.field
            }
          }
        }
      };
    }
    if (indicator.aggregation === 'range' && (indicator.from == null || indicator.to == null)) {
      throw new Error('Invalid Range: both "from" or "to" are required for a range aggregation.');
    }
    if (indicator.aggregation === 'range' && indicator.from != null && indicator.to != null && indicator.from >= indicator.to) {
      throw new Error('Invalid Range: "from" should be less that "to".');
    }
    return {
      filter,
      aggs: {
        total: {
          range: {
            field: indicator.field,
            keyed: true,
            ranges: [{
              key: 'target',
              from: indicator.from,
              to: indicator.to
            }]
          }
        }
      }
    };
  }
  buildBucketScript(type, indicator) {
    if (indicator.aggregation === 'value_count') {
      return {
        bucket_script: {
          buckets_path: {
            value: `_${type}>total`
          },
          script: 'params.value'
        }
      };
    }
    return {
      bucket_script: {
        buckets_path: {
          value: `_${type}>total['target']>_count`
        },
        script: 'params.value'
      }
    };
  }
  execute({
    type,
    aggregationKey
  }) {
    const indicatorDef = this.indicator.params[type];
    return {
      [`_${type}`]: this.buildAggregation(indicatorDef),
      [aggregationKey]: this.buildBucketScript(type, indicatorDef)
    };
  }
}
exports.GetHistogramIndicatorAggregation = GetHistogramIndicatorAggregation;