"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.VERSION = exports.TempSummaryCleanupTask = exports.TYPE = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _elasticsearch = require("@elastic/elasticsearch");
var _task = require("@kbn/task-manager-plugin/server/task");
var _clean_up_temp_summary = require("../management/clean_up_temp_summary");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TYPE = exports.TYPE = 'slo:temp-summary-cleanup-task';
const VERSION = exports.VERSION = '1.0.0';
class TempSummaryCleanupTask {
  constructor(setupContract) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "wasStarted", false);
    const {
      core,
      config,
      taskManager,
      logFactory
    } = setupContract;
    this.logger = logFactory.get(this.taskId);
    this.config = config;
    this.logger.debug('Registering task with [2m] timeout');
    taskManager.registerTaskDefinitions({
      [TYPE]: {
        title: 'SLO temp summary cleanup task',
        timeout: '2m',
        maxAttempts: 1,
        createTaskRunner: ({
          taskInstance,
          abortController
        }) => {
          return {
            run: async () => {
              return this.runTask(taskInstance, core, abortController);
            },
            cancel: async () => {}
          };
        }
      }
    });
  }
  async start(plugins) {
    const hasCorrectLicense = (await plugins.licensing.getLicense()).hasAtLeast('platinum');
    if (!hasCorrectLicense) {
      this.logger.debug('Platinum license is required');
      return;
    }
    if (!plugins.taskManager) {
      this.logger.debug('Missing required service during start');
      return;
    }
    if (!this.config.tempSummaryCleanupTaskEnabled) {
      this.logger.debug('Unscheduling task');
      return await plugins.taskManager.removeIfExists(this.taskId);
    }
    this.logger.debug('Scheduling task with [1h] interval');
    this.wasStarted = true;
    try {
      await plugins.taskManager.ensureScheduled({
        id: this.taskId,
        taskType: TYPE,
        scope: ['observability', 'slo'],
        schedule: {
          interval: '1h'
        },
        state: {},
        params: {}
      });
    } catch (e) {
      this.logger.error(`Error scheduling task, error: ${e}`);
    }
  }
  get taskId() {
    return `${TYPE}:${VERSION}`;
  }
  async runTask(taskInstance, core, abortController) {
    if (!this.wasStarted) {
      this.logger.debug('runTask Aborted. Task not started yet');
      return;
    }
    if (taskInstance.id !== this.taskId) {
      this.logger.debug(`Outdated task version: Got [${taskInstance.id}], current version is [${this.taskId}]`);
      return (0, _task.getDeleteTaskRunResult)();
    }
    this.logger.debug(`runTask started`);
    const [coreStart] = await core.getStartServices();
    const esClient = coreStart.elasticsearch.client.asInternalUser;
    try {
      const cleanUpTempSummary = new _clean_up_temp_summary.CleanUpTempSummary(esClient, this.logger, abortController);
      await cleanUpTempSummary.execute();
    } catch (err) {
      if (err instanceof _elasticsearch.errors.RequestAbortedError) {
        this.logger.warn(`Request aborted due to timeout: ${err}`);
        return;
      }
      this.logger.debug(`Error: ${err}`);
    }
  }
}
exports.TempSummaryCleanupTask = TempSummaryCleanupTask;