"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.executor = executor;
exports.getChecksum = getChecksum;
exports.getInvalidComparatorError = getInvalidComparatorError;
exports.getValidTimefieldSort = getValidTimefieldSort;
exports.tryToParseAsDate = tryToParseAsDate;
var _jsSha = require("js-sha256");
var _i18n = require("@kbn/i18n");
var _alertingRuleUtils = require("@kbn/alerting-rule-utils");
var _common = require("@kbn/triggers-actions-ui-plugin/common");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _server = require("@kbn/alerting-plugin/server");
var _common2 = require("@kbn/response-ops-rule-params/common");
var _objectUtils = require("@kbn/object-utils");
var _action_context = require("./action_context");
var _es_query = require("../../../common/es_query");
var _fetch_es_query = require("./lib/fetch_es_query");
var _fetch_search_source_query = require("./lib/fetch_search_source_query");
var _util = require("./util");
var _fetch_esql_query = require("./lib/fetch_esql_query");
var _ = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function executor(core, options, sourceFields) {
  var _getRecoveredAlerts;
  const searchSourceRule = (0, _util.isSearchSourceRule)(options.params.searchType);
  const esqlQueryRule = (0, _util.isEsqlQueryRule)(options.params.searchType);
  const {
    rule: {
      id: ruleId,
      name
    },
    services,
    params,
    state,
    spaceId,
    logger,
    getTimeRange
  } = options;
  const {
    alertsClient,
    ruleResultService,
    scopedClusterClient,
    share
  } = services;
  if (!alertsClient) {
    throw new _server.AlertsClientError();
  }
  const currentTimestamp = new Date().toISOString();
  const spacePrefix = spaceId !== 'default' ? spaceId : '';
  const alertLimit = alertsClient.getAlertLimitValue();
  const compareFn = _common2.ComparatorFns.get(params.thresholdComparator);
  if (compareFn == null) {
    throw new Error(getInvalidComparatorError(params.thresholdComparator));
  }
  const isGroupAgg = (0, _common.isGroupAggregation)(params.termField) || esqlQueryRule && (0, _common.isPerRowAggregation)(params.groupBy);
  // For ungrouped queries, we run the configured query during each rule run, get a hit count
  // and retrieve up to params.size hits. We evaluate the threshold condition using the
  // value of the hit count. If the threshold condition is met, the hits are counted
  // toward the query match and we update the rule state with the timestamp of the latest hit.
  // In the next run of the rule, the latestTimestamp will be used to gate the query in order to
  // avoid counting a document multiple times.
  // latestTimestamp will be ignored if set for grouped queries
  let latestTimestamp = tryToParseAsDate(state.latestTimestamp);
  const {
    dateStart,
    dateEnd
  } = getTimeRange(`${params.timeWindowSize}${params.timeWindowUnit}`);
  const {
    parsedResults,
    link,
    index
  } = searchSourceRule ? await (0, _fetch_search_source_query.fetchSearchSourceQuery)({
    ruleId,
    alertLimit,
    params: params,
    latestTimestamp,
    spacePrefix,
    services: {
      share,
      getSearchSourceClient: services.getSearchSourceClient,
      logger,
      getDataViews: services.getDataViews,
      ruleResultService
    },
    dateStart,
    dateEnd,
    sourceFields
  }) : esqlQueryRule ? await (0, _fetch_esql_query.fetchEsqlQuery)({
    ruleId,
    alertLimit,
    params: params,
    spacePrefix,
    services: {
      share,
      scopedClusterClient,
      logger,
      ruleResultService
    },
    dateStart,
    dateEnd,
    sourceFields
  }) : await (0, _fetch_es_query.fetchEsQuery)({
    ruleId,
    name,
    alertLimit,
    params: params,
    timestamp: latestTimestamp,
    spacePrefix,
    services: {
      share,
      scopedClusterClient,
      logger,
      ruleResultService
    },
    dateStart,
    dateEnd,
    sourceFields
  });
  const unmetGroupValues = {};
  for (const result of parsedResults.results) {
    var _result$value, _params$threshold;
    const groupingObject = result.groupingObject ? (0, _objectUtils.unflattenObject)(result.groupingObject) : undefined;
    const alertId = result.group;
    const value = (_result$value = result.value) !== null && _result$value !== void 0 ? _result$value : result.count;

    // group aggregations use the bucket selector agg to compare conditions
    // within the ES query, so only 'met' results are returned, therefore we don't need
    // to use the compareFn
    const met = isGroupAgg ? true : compareFn(value, params.threshold);
    if (!met) {
      unmetGroupValues[alertId] = value;
      continue;
    }
    const baseContext = {
      title: name,
      date: currentTimestamp,
      value,
      hits: result.hits,
      link,
      sourceFields: result.sourceFields,
      grouping: groupingObject
    };
    const baseActiveContext = {
      ...baseContext,
      conditions: (0, _action_context.getContextConditionsDescription)({
        searchType: params.searchType,
        comparator: params.thresholdComparator,
        threshold: params.threshold,
        aggType: params.aggType,
        aggField: params.aggField,
        ...(isGroupAgg ? {
          group: alertId
        } : {})
      })
    };
    const actionContext = (0, _action_context.addMessages)({
      ruleName: name,
      baseContext: baseActiveContext,
      params,
      ...(isGroupAgg ? {
        group: alertId
      } : {}),
      index
    });
    const id = alertId === _common.UngroupedGroupId && !isGroupAgg ? _es_query.ConditionMetAlertInstanceId : alertId;
    const ecsGroups = (0, _alertingRuleUtils.getEcsGroupsFromFlattenGrouping)(result.groupingObject);
    alertsClient.report({
      id,
      actionGroup: _es_query.ActionGroupId,
      state: {
        latestTimestamp,
        dateStart,
        dateEnd
      },
      context: actionContext,
      payload: {
        [_ruleDataUtils.ALERT_URL]: actionContext.link,
        [_ruleDataUtils.ALERT_REASON]: actionContext.message,
        [_.ALERT_TITLE]: actionContext.title,
        [_.ALERT_EVALUATION_CONDITIONS]: actionContext.conditions,
        [_ruleDataUtils.ALERT_EVALUATION_VALUE]: `${actionContext.value}`,
        [_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]: ((_params$threshold = params.threshold) === null || _params$threshold === void 0 ? void 0 : _params$threshold.length) === 1 ? params.threshold[0] : null,
        [_ruleDataUtils.ALERT_GROUPING]: groupingObject,
        ...ecsGroups,
        ...actionContext.sourceFields
      }
    });
    if (!isGroupAgg) {
      var _result$hits$find;
      // update the timestamp based on the current search results
      const firstValidTimefieldSort = getValidTimefieldSort(// @ts-expect-error `sort` now depends on `FieldValue` that is too broad
      (_result$hits$find = result.hits.find(hit => getValidTimefieldSort(hit.sort))) === null || _result$hits$find === void 0 ? void 0 : _result$hits$find.sort);
      if (firstValidTimefieldSort) {
        latestTimestamp = firstValidTimefieldSort;
      }
    }
  }
  alertsClient.setAlertLimitReached(parsedResults.truncated);
  const {
    getRecoveredAlerts
  } = alertsClient;
  const recoveredAlerts = (_getRecoveredAlerts = getRecoveredAlerts()) !== null && _getRecoveredAlerts !== void 0 ? _getRecoveredAlerts : [];
  for (const recoveredAlert of recoveredAlerts) {
    var _unmetGroupValues$ale, _recoveredAlert$hit, _params$threshold2;
    const alertId = recoveredAlert.alert.getId();
    const baseRecoveryContext = {
      title: name,
      date: currentTimestamp,
      value: (_unmetGroupValues$ale = unmetGroupValues[alertId]) !== null && _unmetGroupValues$ale !== void 0 ? _unmetGroupValues$ale : 0,
      hits: [],
      link,
      conditions: (0, _action_context.getContextConditionsDescription)({
        searchType: params.searchType,
        comparator: params.thresholdComparator,
        threshold: params.threshold,
        isRecovered: true,
        aggType: params.aggType,
        aggField: params.aggField,
        ...(isGroupAgg ? {
          group: alertId
        } : {})
      }),
      sourceFields: [],
      grouping: (_recoveredAlert$hit = recoveredAlert.hit) === null || _recoveredAlert$hit === void 0 ? void 0 : _recoveredAlert$hit[_ruleDataUtils.ALERT_GROUPING]
    };
    const recoveryContext = (0, _action_context.addMessages)({
      ruleName: name,
      baseContext: baseRecoveryContext,
      params,
      isRecovered: true,
      ...(isGroupAgg ? {
        group: alertId
      } : {}),
      index
    });
    alertsClient.setAlertData({
      id: alertId,
      context: recoveryContext,
      payload: {
        [_ruleDataUtils.ALERT_URL]: recoveryContext.link,
        [_ruleDataUtils.ALERT_REASON]: recoveryContext.message,
        [_.ALERT_TITLE]: recoveryContext.title,
        [_.ALERT_EVALUATION_CONDITIONS]: recoveryContext.conditions,
        [_ruleDataUtils.ALERT_EVALUATION_VALUE]: `${recoveryContext.value}`,
        [_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]: ((_params$threshold2 = params.threshold) === null || _params$threshold2 === void 0 ? void 0 : _params$threshold2.length) === 1 ? params.threshold[0] : null
      }
    });
  }
  return {
    state: {
      latestTimestamp
    }
  };
}
function getValidTimefieldSort(sortValues = []) {
  for (const sortValue of sortValues) {
    const sortDate = tryToParseAsDate(sortValue);
    if (sortDate) {
      return sortDate;
    }
  }
}
function tryToParseAsDate(sortValue) {
  const sortDate = typeof sortValue === 'string' ? Date.parse(sortValue) : sortValue;
  if (sortDate && !isNaN(sortDate)) {
    return new Date(sortDate).toISOString();
  }
}
function getChecksum(params) {
  return _jsSha.sha256.create().update(JSON.stringify(params));
}
function getInvalidComparatorError(comparator) {
  return _i18n.i18n.translate('xpack.stackAlerts.esQuery.invalidComparatorErrorMessage', {
    defaultMessage: 'invalid thresholdComparator specified: {comparator}',
    values: {
      comparator
    }
  });
}