"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useGetAppInfo = void 0;
var _lodash = require("lodash");
var _react = require("react");
var _api = require("./api");
var _translations = require("./translations");
var _cors_error = require("./cors_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useGetAppInfo = ({
  actionTypeId,
  http
}) => {
  const [isLoading, setIsLoading] = (0, _react.useState)(false);
  const didCancel = (0, _react.useRef)(false);
  const abortCtrl = (0, _react.useRef)(new AbortController());
  const fetchAppInfo = (0, _react.useCallback)(async connector => {
    try {
      if (!actionTypeId || (0, _lodash.isEmpty)(actionTypeId)) {
        return;
      }
      didCancel.current = false;
      abortCtrl.current.abort();
      abortCtrl.current = new AbortController();
      setIsLoading(true);
      const res = await (0, _api.getAppInfo)({
        signal: abortCtrl.current.signal,
        connector,
        actionTypeId,
        http
      });
      if (!didCancel.current) {
        setIsLoading(false);
      }
      return res;
    } catch (error) {
      if (!didCancel.current) {
        setIsLoading(false);
      }

      /**
       * According to https://developer.mozilla.org/en-US/docs/Web/API/fetch#exceptions
       * all network errors throw a TypeError. Usually fetch errors are happening
       * due to CORS misconfiguration. By detecting fetch errors we can provide
       * a better message about CORS. Adding a CORS rule to allow requests from the UI
       * in the ServiceNow instance is needed by our ServiceNow applications.
       */
      if (error.name === 'TypeError') {
        throw new _cors_error.CORSError(_translations.FETCH_ERROR);
      }
      throw error;
    }
  }, [actionTypeId, http]);
  (0, _react.useEffect)(() => {
    return () => {
      didCancel.current = true;
      abortCtrl.current.abort();
      setIsLoading(false);
    };
  }, []);
  return {
    fetchAppInfo,
    isLoading
  };
};
exports.useGetAppInfo = useGetAppInfo;