"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.identifyFeatures = identifyFeatures;
var _aiTools = require("@kbn/ai-tools");
var _inferencePromptUtils = require("@kbn/inference-prompt-utils");
var _pLimit = _interopRequireDefault(require("p-limit"));
var _prompt = require("./prompt");
var _cluster_logs = require("../cluster_logs/cluster_logs");
var _condition_schema = _interopRequireDefault(require("../shared/condition_schema.text"));
var _generate_description = require("../description/generate_description");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CONCURRENT_DESCRIPTION_REQUESTS = 5;

/**
 * Identifies features in a stream, by:
 * - describing the dataset (via sampled documents)
 * - clustering docs together on similarity
 * - asking the LLM to identify features by creating
 * queries and validating the resulting clusters
 */
async function identifyFeatures({
  stream,
  features,
  start,
  end,
  esClient,
  kql,
  inferenceClient,
  logger,
  dropUnmapped = false
}) {
  var _features$map;
  const [analysis, initialClustering] = await Promise.all([(0, _aiTools.describeDataset)({
    start,
    end,
    esClient,
    index: stream.name,
    kql: kql || undefined
  }), (0, _cluster_logs.clusterLogs)({
    start,
    end,
    esClient,
    index: stream.name,
    partitions: (_features$map = features === null || features === void 0 ? void 0 : features.map(feature => {
      return {
        name: feature.name,
        condition: feature.filter
      };
    })) !== null && _features$map !== void 0 ? _features$map : [],
    logger,
    dropUnmapped
  })]);
  const response = await (0, _inferencePromptUtils.executeAsReasoningAgent)({
    maxSteps: 3,
    input: {
      stream: {
        name: stream.name,
        description: stream.description || 'This stream has no description.'
      },
      dataset_analysis: JSON.stringify((0, _aiTools.sortAndTruncateAnalyzedFields)(analysis, {
        dropEmpty: true,
        dropUnmapped
      })),
      initial_clustering: JSON.stringify(initialClustering),
      condition_schema: _condition_schema.default
    },
    prompt: _prompt.IdentifySystemsPrompt,
    inferenceClient,
    finalToolChoice: {
      function: 'finalize_systems'
    },
    toolCallbacks: {
      validate_systems: async toolCall => {
        const clustering = await (0, _cluster_logs.clusterLogs)({
          start,
          end,
          esClient,
          index: stream.name,
          logger,
          partitions: toolCall.function.arguments.systems.map(system => {
            return {
              name: system.name,
              condition: system.filter
            };
          }),
          dropUnmapped
        });
        return {
          response: {
            systems: clustering.map(cluster => {
              return {
                name: cluster.name,
                clustering: cluster.clustering
              };
            })
          }
        };
      },
      finalize_systems: async toolCall => {
        return {
          response: {}
        };
      }
    }
  });
  const limiter = (0, _pLimit.default)(CONCURRENT_DESCRIPTION_REQUESTS);
  return {
    features: await Promise.all(response.toolCalls.flatMap(toolCall => toolCall.function.arguments.systems.map(async args => {
      const feature = {
        ...args,
        filter: args.filter,
        description: ''
      };
      const description = await limiter(async () => {
        return await (0, _generate_description.generateStreamDescription)({
          stream,
          start,
          end,
          esClient,
          inferenceClient,
          feature
        });
      });
      return {
        ...feature,
        description
      };
    })))
  };
}