"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPanelDimensions = void 0;
const d3_scale_1 = require("d3-scale");
const get_bullet_spec_1 = require("./get_bullet_spec");
const get_layout_1 = require("./get_layout");
const create_selector_1 = require("../../../state/create_selector");
const get_chart_theme_1 = require("../../../state/selectors/get_chart_theme");
const get_resolved_background_color_1 = require("../../../state/selectors/get_resolved_background_color");
const common_1 = require("../../../utils/common");
const constants_1 = require("../renderer/canvas/constants");
const spec_1 = require("../spec");
const theme_1 = require("../theme");
const angular_1 = require("../utils/angular");
const color_1 = require("../utils/color");
const ticks_1 = require("../utils/ticks");
exports.getPanelDimensions = (0, create_selector_1.createCustomCachedSelector)([get_layout_1.getLayout, get_bullet_spec_1.getBulletSpec, get_chart_theme_1.getChartThemeSelector, get_resolved_background_color_1.getResolvedBackgroundColorSelector], ({ shouldRenderMetric, headerLayout, layoutAlignment, panel: panelSize }, spec, { bulletGraph: bulletGraphStyles }, backgroundColor) => {
    const rows = headerLayout.map((row, rowIndex) => {
        return row.map((bulletGraph, columnIndex) => {
            if (!bulletGraph)
                return null;
            const { panel, multiline, datum, ...rest } = bulletGraph;
            const verticalAlignment = layoutAlignment[rowIndex];
            const graphSize = {
                width: panel.width,
                height: panel.height - verticalAlignment.headerHeight,
            };
            return {
                ...rest,
                ...getSubtypeDimensions(spec, graphSize, datum, bulletGraphStyles, backgroundColor),
                datum,
                multiline,
                graphArea: {
                    size: graphSize,
                    origin: {
                        x: 0,
                        y: panel.height - graphSize.height,
                    },
                    center: {
                        x: graphSize.width / 2 - theme_1.GRAPH_PADDING.left,
                        y: graphSize.height / 2 - theme_1.GRAPH_PADDING.top,
                    },
                },
                panel: {
                    x: panel.width * columnIndex,
                    y: panel.height * rowIndex,
                    ...panel,
                },
            };
        });
    });
    return {
        rows,
        panel: panelSize,
        layoutAlignment,
        shouldRenderMetric,
    };
});
function getSubtypeDimensions({ subtype, colorBands: colorBandsConfig }, graphSize, { ticks: desiredTicks, domain, niceDomain }, { colorBands: defaultColorBandsConfig, fallbackBandColor }, backgroundColor) {
    switch (subtype) {
        case spec_1.BulletSubtype.circle:
        case spec_1.BulletSubtype.halfCircle:
        case spec_1.BulletSubtype.twoThirdsCircle: {
            const [startAngle, endAngle] = (0, angular_1.getAnglesBySize)(subtype);
            const { radius } = (0, angular_1.getAngledChartSizing)(graphSize, subtype);
            const { scale, ticks } = getScaleWithTicks(domain, [startAngle, endAngle], {
                rangeMultiplier: radius,
                desiredTicks,
                nice: niceDomain,
                interval: constants_1.ANGULAR_TICK_INTERVAL,
            });
            const { bands: colorBands, scale: colorScale } = (0, color_1.getColorScaleWithBands)(scale, colorBandsConfig ?? defaultColorBandsConfig, ticks, backgroundColor, fallbackBandColor);
            return {
                scale,
                domain: scale.domain(),
                ticks,
                colorBands,
                colorScale,
            };
        }
        case spec_1.BulletSubtype.horizontal: {
            const paddedWidth = graphSize.width - theme_1.GRAPH_PADDING.left - theme_1.GRAPH_PADDING.right;
            const { scale, ticks } = getScaleWithTicks(domain, [0, paddedWidth], {
                desiredTicks,
                nice: niceDomain,
                interval: constants_1.TICK_INTERVAL,
            });
            const { bands: colorBands, scale: colorScale } = (0, color_1.getColorScaleWithBands)(scale, colorBandsConfig ?? defaultColorBandsConfig, ticks, backgroundColor, fallbackBandColor);
            return {
                scale,
                domain: scale.domain(),
                ticks,
                colorBands,
                colorScale,
            };
        }
        case spec_1.BulletSubtype.vertical: {
            const paddedHeight = graphSize.height - theme_1.GRAPH_PADDING.bottom - theme_1.GRAPH_PADDING.top;
            const { scale, ticks } = getScaleWithTicks(domain, [0, paddedHeight], {
                desiredTicks,
                nice: niceDomain,
                interval: constants_1.TICK_INTERVAL,
            });
            const { bands: colorBands, scale: colorScale } = (0, color_1.getColorScaleWithBands)(scale, colorBandsConfig ?? defaultColorBandsConfig, ticks, backgroundColor, fallbackBandColor);
            return {
                scale,
                domain: scale.domain(),
                ticks,
                colorBands,
                colorScale,
            };
        }
        default:
            throw new Error('Unknown Bullet subtype');
    }
}
function getScaleWithTicks(domain, range, tickOptions) {
    let scale = (0, d3_scale_1.scaleLinear)().domain(domain).range(range);
    const scaleRange = scale.range();
    const ticks = (0, ticks_1.getTicks)(Math.abs(scaleRange[1] - scaleRange[0]) * (tickOptions.rangeMultiplier || 1), tickOptions);
    const customRange = typeof ticks !== 'number';
    if (tickOptions.nice) {
        scale = scale.nice(customRange ? undefined : ticks);
    }
    const updatedDomain = scale.domain();
    return {
        scale,
        ticks: customRange
            ? (Array.isArray(ticks) ? ticks : ticks(updatedDomain)).filter((0, common_1.isWithinRange)(updatedDomain))
            : scale.ticks(ticks),
    };
}
//# sourceMappingURL=get_panel_dimensions.js.map