"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handleTooltipActions = exports.handleDOMElementActions = exports.handleLegendActions = exports.handleMouseActions = exports.handleKeyActions = void 0;
const immer_1 = require("immer");
const chart_types_1 = require("../../chart_types");
const drilldown_active_1 = require("../../chart_types/partition_chart/state/selectors/drilldown_active");
const picked_shapes_1 = require("../../chart_types/partition_chart/state/selectors/picked_shapes");
const keys_1 = require("../../utils/keys");
const point_1 = require("../../utils/point");
const dom_element_1 = require("../actions/dom_element");
const key_1 = require("../actions/key");
const legend_1 = require("../actions/legend");
const mouse_1 = require("../actions/mouse");
const tooltip_1 = require("../actions/tooltip");
const get_internal_is_intialized_1 = require("../selectors/get_internal_is_intialized");
const get_internal_is_tooltip_visible_1 = require("../selectors/get_internal_is_tooltip_visible");
const get_internal_tooltip_info_1 = require("../selectors/get_internal_tooltip_info");
const get_legend_items_1 = require("../selectors/get_legend_items");
const get_tooltip_spec_1 = require("../selectors/get_tooltip_spec");
const get_initial_pointer_state_1 = require("../utils/get_initial_pointer_state");
const get_initial_tooltip_state_1 = require("../utils/get_initial_tooltip_state");
function createItemId(item) {
    return `${item.seriesIdentifier.key}-${item.label}-${item.value}`;
}
const DRAG_DETECTION_PIXEL_DELTA = 4;
const handleKeyActions = (builder) => {
    builder.addCase(key_1.onKeyPress, (globalState, action) => {
        if ((0, get_internal_is_intialized_1.getInternalIsInitializedSelector)(globalState) !== get_internal_is_intialized_1.InitStatus.Initialized)
            return;
        const state = globalState.interactions;
        if (action.payload === 'Escape') {
            if (state.tooltip.pinned) {
                state.pointer = (0, get_initial_pointer_state_1.getInitialPointerState)();
                state.tooltip = (0, get_initial_tooltip_state_1.getInitialTooltipState)();
                return;
            }
            state.pointer = (0, get_initial_pointer_state_1.getInitialPointerState)();
        }
    });
};
exports.handleKeyActions = handleKeyActions;
const handleMouseActions = (builder) => {
    builder
        .addCase(mouse_1.onPointerMove, (globalState, action) => {
        if ((0, get_internal_is_intialized_1.getInternalIsInitializedSelector)(globalState) !== get_internal_is_intialized_1.InitStatus.Initialized)
            return;
        const state = globalState.interactions;
        const dragging = state.pointer.dragging ||
            (!!state.pointer.down &&
                (0, point_1.getDelta)(state.pointer.down.position, action.payload.position) > DRAG_DETECTION_PIXEL_DELTA);
        state.pointer.dragging = dragging;
        state.pointer.current.position = action.payload.position;
        state.pointer.current.time = action.payload.time;
    })
        .addCase(mouse_1.onMouseDown, (globalState, action) => {
        if ((0, get_internal_is_intialized_1.getInternalIsInitializedSelector)(globalState) !== get_internal_is_intialized_1.InitStatus.Initialized)
            return;
        const state = globalState.interactions;
        state.prevDrilldown = state.drilldown;
        state.drilldown = getDrilldownData(globalState);
        state.pointer.dragging = false;
        state.pointer.up = null;
        state.pointer.down = {
            position: action.payload.position,
            time: action.payload.time,
        };
        state.pointer.keyPressed = action.payload.keyPressed ?? keys_1.noModifierKeysPressed;
    })
        .addCase(mouse_1.onMouseUp, (globalState, action) => {
        if ((0, get_internal_is_intialized_1.getInternalIsInitializedSelector)(globalState) !== get_internal_is_intialized_1.InitStatus.Initialized)
            return;
        const state = globalState.interactions;
        state.pointer.lastDrag =
            state.pointer.down && state.pointer.dragging
                ? {
                    start: {
                        position: {
                            ...state.pointer.down.position,
                        },
                        time: state.pointer.down.time,
                    },
                    end: {
                        position: {
                            ...state.pointer.current.position,
                        },
                        time: action.payload.time,
                    },
                }
                : null;
        state.pointer.lastClick =
            state.pointer.down && !state.pointer.dragging
                ? {
                    position: {
                        ...action.payload.position,
                    },
                    time: action.payload.time,
                }
                : null;
        state.pointer.dragging = false;
        state.pointer.down = null;
        state.pointer.up = {
            position: action.payload.position,
            time: action.payload.time,
        };
    });
};
exports.handleMouseActions = handleMouseActions;
const handleLegendActions = (builder) => {
    builder.addCase(legend_1.onLegendItemOutAction, (globalState) => {
        if ((0, get_internal_is_intialized_1.getInternalIsInitializedSelector)(globalState) !== get_internal_is_intialized_1.InitStatus.Initialized)
            return;
        const state = globalState.interactions;
        state.highlightedLegendPath = [];
    });
    builder.addCase(legend_1.onLegendItemOverAction, (globalState, action) => {
        if ((0, get_internal_is_intialized_1.getInternalIsInitializedSelector)(globalState) !== get_internal_is_intialized_1.InitStatus.Initialized)
            return;
        const state = globalState.interactions;
        state.highlightedLegendPath = action.payload;
    });
    builder.addCase(legend_1.onToggleDeselectSeriesAction, (globalState, action) => {
        if ((0, get_internal_is_intialized_1.getInternalIsInitializedSelector)(globalState) !== get_internal_is_intialized_1.InitStatus.Initialized)
            return;
        const state = globalState.interactions;
        state.deselectedDataSeries = toggleDeselectedDataSeries(action.payload, state.deselectedDataSeries, (0, get_legend_items_1.getLegendItemsSelector)(globalState));
    });
};
exports.handleLegendActions = handleLegendActions;
const handleDOMElementActions = (builder) => {
    builder
        .addCase(dom_element_1.onDOMElementEnter, (globalState, action) => {
        if ((0, get_internal_is_intialized_1.getInternalIsInitializedSelector)(globalState) !== get_internal_is_intialized_1.InitStatus.Initialized)
            return;
        const state = globalState.interactions;
        state.hoveredDOMElement = action.payload;
    })
        .addCase(dom_element_1.onDOMElementLeave, (globalState) => {
        if ((0, get_internal_is_intialized_1.getInternalIsInitializedSelector)(globalState) !== get_internal_is_intialized_1.InitStatus.Initialized)
            return;
        const state = globalState.interactions;
        state.hoveredDOMElement = null;
    });
};
exports.handleDOMElementActions = handleDOMElementActions;
const handleTooltipActions = (builder) => {
    builder
        .addCase(tooltip_1.pinTooltip, (globalState, action) => {
        if ((0, get_internal_is_intialized_1.getInternalIsInitializedSelector)(globalState) !== get_internal_is_intialized_1.InitStatus.Initialized)
            return;
        const state = globalState.interactions;
        if (!action.payload.pinned) {
            if (action.payload.resetPointer) {
                state.pointer = (0, get_initial_pointer_state_1.getInitialPointerState)();
            }
            else {
                state.pointer.pinned = null;
            }
            state.tooltip = (0, get_initial_tooltip_state_1.getInitialTooltipState)();
            return;
        }
        const { isPinnable, displayOnly } = (0, get_internal_is_tooltip_visible_1.getInternalIsTooltipVisibleSelector)(globalState);
        if (!isPinnable || displayOnly) {
            return;
        }
        const tooltipSpec = (0, get_tooltip_spec_1.getTooltipSpecSelector)(globalState);
        const getSelectedValues = () => {
            const values = (0, get_internal_tooltip_info_1.getInternalTooltipInfoSelector)(globalState)?.values ?? [];
            if (globalState.chartType === chart_types_1.ChartType.Heatmap)
                return values.slice(0, 1);
            return values.filter((v) => globalState.chartType === chart_types_1.ChartType.XYAxis ? v.isHighlighted : !v.displayOnly);
        };
        const selected = Array.isArray(tooltipSpec.actions) && tooltipSpec.actions.length === 0 ? [] : getSelectedValues();
        state.tooltip.pinned = true;
        state.tooltip.selected = selected;
        state.pointer.pinned = state.pointer.current;
    })
        .addCase(tooltip_1.toggleSelectedTooltipItem, (globalState, action) => {
        if ((0, get_internal_is_intialized_1.getInternalIsInitializedSelector)(globalState) !== get_internal_is_intialized_1.InitStatus.Initialized)
            return;
        const state = globalState.interactions;
        if (!state.tooltip.pinned)
            return;
        const index = state.tooltip.selected.findIndex((item) => createItemId(item) === createItemId(action.payload));
        if (index !== -1) {
            state.tooltip.selected = (0, immer_1.produce)(state.tooltip.selected, (draft) => {
                draft.splice(index, 1);
            });
        }
        else {
            state.tooltip.selected.push(action.payload);
        }
    })
        .addCase(tooltip_1.setSelectedTooltipItems, (globalState, action) => {
        if ((0, get_internal_is_intialized_1.getInternalIsInitializedSelector)(globalState) !== get_internal_is_intialized_1.InitStatus.Initialized)
            return;
        const state = globalState.interactions;
        if (!state.tooltip.pinned)
            return;
        state.tooltip.selected = action.payload;
    });
};
exports.handleTooltipActions = handleTooltipActions;
function toggleDeselectedDataSeries({ legendItemIds, metaKey }, deselectedDataSeries, legendItems) {
    const actionSeriesKeys = legendItemIds.map(({ key }) => key);
    const deselectedDataSeriesKeys = new Set(deselectedDataSeries.map(({ key }) => key));
    const legendItemsKeys = legendItems.map(({ seriesIdentifiers }) => seriesIdentifiers);
    const alreadyDeselected = actionSeriesKeys.every((key) => deselectedDataSeriesKeys.has(key));
    const keepOnlyNonActionSeries = ({ key }) => !actionSeriesKeys.includes(key);
    if (metaKey) {
        return alreadyDeselected
            ? deselectedDataSeries.filter(keepOnlyNonActionSeries)
            : deselectedDataSeries.concat(legendItemIds);
    }
    if (alreadyDeselected) {
        return deselectedDataSeries.filter(keepOnlyNonActionSeries);
    }
    if (deselectedDataSeries.length === legendItemsKeys.length - 1) {
        return [];
    }
    return deselectedDataSeries.length
        ? deselectedDataSeries.concat(legendItemIds)
        : legendItemsKeys.flat().filter(keepOnlyNonActionSeries);
}
function getDrilldownData(globalState) {
    if (globalState.chartType !== chart_types_1.ChartType.Partition || !(0, drilldown_active_1.drilldownActive)(globalState)) {
        return [];
    }
    const layerValues = (0, picked_shapes_1.getPickedShapesLayerValues)(globalState)[0];
    return layerValues ? layerValues.at(-1)?.path.map((n) => n.value) ?? [] : [];
}
//# sourceMappingURL=interactions.js.map