"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useFlyoutLayoutMode = exports.useApplyFlyoutLayoutMode = exports.getWidthFromSize = void 0;
var _react = require("react");
var _services = require("../../../services");
var _actions = require("./actions");
var _selectors = require("./selectors");
var _hooks = require("./hooks");
var _const = require("./const");
function _slicedToArray(r, e) { return _arrayWithHoles(r) || _iterableToArrayLimit(r, e) || _unsupportedIterableToArray(r, e) || _nonIterableRest(); }
function _nonIterableRest() { throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }
function _unsupportedIterableToArray(r, a) { if (r) { if ("string" == typeof r) return _arrayLikeToArray(r, a); var t = {}.toString.call(r).slice(8, -1); return "Object" === t && r.constructor && (t = r.constructor.name), "Map" === t || "Set" === t ? Array.from(r) : "Arguments" === t || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(t) ? _arrayLikeToArray(r, a) : void 0; } }
function _arrayLikeToArray(r, a) { (null == a || a > r.length) && (a = r.length); for (var e = 0, n = Array(a); e < a; e++) n[e] = r[e]; return n; }
function _iterableToArrayLimit(r, l) { var t = null == r ? null : "undefined" != typeof Symbol && r[Symbol.iterator] || r["@@iterator"]; if (null != t) { var e, n, i, u, a = [], f = !0, o = !1; try { if (i = (t = t.call(r)).next, 0 === l) { if (Object(t) !== t) return; f = !1; } else for (; !(f = (e = i.call(t)).done) && (a.push(e.value), a.length !== l); f = !0); } catch (r) { o = !0, n = r; } finally { try { if (!f && null != t.return && (u = t.return(), Object(u) !== u)) return; } finally { if (o) throw n; } } return a; } }
function _arrayWithHoles(r) { if (Array.isArray(r)) return r; } /*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
/**
 * Hook to handle responsive layout mode for managed flyouts.
 * Decides whether to place flyouts side-by-side or stacked based on
 * viewport width and flyout widths/sizes.
 */
var useApplyFlyoutLayoutMode = exports.useApplyFlyoutLayoutMode = function useApplyFlyoutLayoutMode() {
  var _context$state;
  var _useEuiTheme = (0, _services.useEuiTheme)(),
    euiTheme = _useEuiTheme.euiTheme;
  var context = (0, _hooks.useFlyoutManager)();
  var currentSession = (0, _selectors.useCurrentSession)();
  var parentFlyoutId = currentSession === null || currentSession === void 0 ? void 0 : currentSession.mainFlyoutId;
  var childFlyoutId = currentSession === null || currentSession === void 0 ? void 0 : currentSession.childFlyoutId;
  var parentFlyout = (0, _selectors.useCurrentMainFlyout)();
  var childFlyout = (0, _selectors.useCurrentChildFlyout)();
  var parentWidth = (0, _selectors.useFlyoutWidth)(parentFlyoutId);
  var childWidth = (0, _selectors.useFlyoutWidth)(childFlyoutId);
  var hasFlyouts = Boolean(parentFlyoutId);
  var _useState = (0, _react.useState)(typeof window !== 'undefined' ? window.innerWidth : Infinity),
    _useState2 = _slicedToArray(_useState, 2),
    windowWidth = _useState2[0],
    setWindowWidth = _useState2[1];

  // Extract specific context values
  var dispatch = context === null || context === void 0 ? void 0 : context.dispatch;
  var currentLayoutMode = context === null || context === void 0 || (_context$state = context.state) === null || _context$state === void 0 ? void 0 : _context$state.layoutMode;
  var setMode = (0, _react.useCallback)(function (layoutMode) {
    if (dispatch) {
      dispatch((0, _actions.setLayoutMode)(layoutMode));
    }
  }, [dispatch]);
  (0, _react.useEffect)(function () {
    if (typeof window === 'undefined') {
      return;
    }
    var rafId = 0;
    var handleResize = function handleResize() {
      if (rafId) {
        cancelAnimationFrame(rafId);
      }
      rafId = requestAnimationFrame(function () {
        return setWindowWidth(window.innerWidth);
      });
    };
    window.addEventListener('resize', handleResize);
    return function () {
      if (rafId) {
        cancelAnimationFrame(rafId);
      }
      window.removeEventListener('resize', handleResize);
    };
  }, []);

  // Calculate the desired layout mode
  var desiredLayoutMode = (0, _react.useMemo)(function () {
    // Skip calculation if no flyouts open
    if (!hasFlyouts) {
      return null;
    }

    // Thresholds to prevent thrashing near the breakpoint.
    var THRESHOLD_TO_SIDE_BY_SIDE = 85;
    var THRESHOLD_TO_STACKED = 95;

    // If the window is too small, set the mode to stacked.
    //
    // The value is based on the maximum width of a flyout in
    // `composeFlyoutSizing` in `flyout.styles.ts` multiplied
    // by 2 (open flyouts side-by-side).
    if (windowWidth < Math.round(euiTheme.breakpoint.s * 1.4)) {
      return _const.LAYOUT_MODE_STACKED;
    }
    if (!childFlyoutId) {
      return _const.LAYOUT_MODE_SIDE_BY_SIDE;
    }
    var parentWidthValue = parentWidth;
    var childWidthValue = childWidth;
    if (!parentWidthValue && parentFlyout !== null && parentFlyout !== void 0 && parentFlyout.size) {
      parentWidthValue = getWidthFromSize(parentFlyout.size);
    }
    if (!childWidthValue && childFlyout !== null && childFlyout !== void 0 && childFlyout.size) {
      childWidthValue = getWidthFromSize(childFlyout.size);
    }
    if (!parentWidthValue || !childWidthValue) {
      return _const.LAYOUT_MODE_SIDE_BY_SIDE;
    }
    var combinedWidth = parentWidthValue + childWidthValue;
    var combinedWidthPercentage = combinedWidth / windowWidth * 100;

    // Handle fill size flyouts: keep layout as side-by-side when fill flyout is present
    // This allows fill flyouts to dynamically calculate their width based on the other in the pair
    if ((parentFlyout === null || parentFlyout === void 0 ? void 0 : parentFlyout.size) === 'fill' || (childFlyout === null || childFlyout === void 0 ? void 0 : childFlyout.size) === 'fill') {
      // For fill flyouts, we want to maintain side-by-side layout to enable dynamic width calculation
      // Only stack if the viewport is too small (below the small breakpoint)
      if (windowWidth >= Math.round(euiTheme.breakpoint.s * 1.4)) {
        return _const.LAYOUT_MODE_SIDE_BY_SIDE;
      }
    }
    if (currentLayoutMode === _const.LAYOUT_MODE_STACKED) {
      return combinedWidthPercentage <= THRESHOLD_TO_SIDE_BY_SIDE ? _const.LAYOUT_MODE_SIDE_BY_SIDE : _const.LAYOUT_MODE_STACKED;
    } else {
      return combinedWidthPercentage >= THRESHOLD_TO_STACKED ? _const.LAYOUT_MODE_STACKED : _const.LAYOUT_MODE_SIDE_BY_SIDE;
    }
  }, [hasFlyouts, windowWidth, euiTheme, childFlyoutId, parentWidth, childWidth, parentFlyout === null || parentFlyout === void 0 ? void 0 : parentFlyout.size, childFlyout === null || childFlyout === void 0 ? void 0 : childFlyout.size, currentLayoutMode]);

  // Apply the desired layout mode
  (0, _react.useEffect)(function () {
    if (desiredLayoutMode && currentLayoutMode !== desiredLayoutMode) {
      setMode(desiredLayoutMode);
    }
  }, [desiredLayoutMode, currentLayoutMode, setMode]);
};

/** Convert a flyout `size` value to a pixel width using theme breakpoints. */
var getWidthFromSize = exports.getWidthFromSize = function getWidthFromSize(size) {
  if (typeof size === 'number') {
    return size;
  }
  if (typeof size === 'string') {
    var parsed = parseInt(size, 10);
    if (!Number.isNaN(parsed)) {
      return parsed;
    }

    // Size is a function of a percentage of `vw`, defined in `composeFlyoutSizing` in `flyout.styles.ts`
    switch (size) {
      case 's':
        return Math.round(window.innerWidth * 0.25);
      case 'm':
        return Math.round(window.innerWidth * 0.5);
      case 'l':
        return Math.round(window.innerWidth * 0.75);
      case 'fill':
        return Math.round(window.innerWidth * 0.9);
      default:
        break;
    }
  }
  return 0;
};

/** Current layout mode for managed flyouts (`side-by-side` or `stacked`). */
var useFlyoutLayoutMode = exports.useFlyoutLayoutMode = function useFlyoutLayoutMode() {
  var _context$state2;
  var context = (0, _hooks.useFlyoutManager)();
  return (context === null || context === void 0 || (_context$state2 = context.state) === null || _context$state2 === void 0 ? void 0 : _context$state2.layoutMode) || _const.LAYOUT_MODE_SIDE_BY_SIDE;
};