"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createValidationErrorMessage = createValidationErrorMessage;
exports.isNamedSize = isNamedSize;
exports.validateManagedFlyoutSize = validateManagedFlyoutSize;
exports.validateSizeCombination = validateSizeCombination;
var _const = require("../const");
var _const2 = require("./const");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Checks if a size is a named size (s, m, l)
 */
function isNamedSize(size) {
  return _const.FLYOUT_SIZES.includes(size);
}

/**
 * Validates that a managed flyout only uses named sizes
 */
function validateManagedFlyoutSize(size, flyoutId, level) {
  if (level === _const2.LEVEL_CHILD && !isNamedSize(size)) {
    var namedSizes = _const.FLYOUT_SIZES.join(', ');
    return {
      type: 'INVALID_SIZE_TYPE',
      message: "Child flyout ".concat(flyoutId, " must use a named size (").concat(namedSizes, "). Received: ").concat(size),
      flyoutId: flyoutId,
      level: level,
      size: size
    };
  }
  return null;
}

/**
 * Validates size combinations for parent-child flyouts
 */
function validateSizeCombination(parentSize, childSize) {
  var sizes = [parentSize, childSize];

  // Parent and child can't both be 'm'
  if (sizes.every(function (s) {
    return s === 'm';
  })) {
    return {
      type: 'INVALID_SIZE_COMBINATION',
      message: 'Parent and child flyouts cannot both be size "m"'
    };
  }

  // Parent and child can't both be 'fill'
  if (sizes.every(function (s) {
    return s === 'fill';
  })) {
    return {
      type: 'INVALID_SIZE_COMBINATION',
      message: 'Parent and child flyouts cannot both be size "fill"'
    };
  }

  // Flyout can't be 'l' if the other in the pair is not "fill"
  if (sizes.includes('l') && !sizes.includes('fill')) {
    return {
      type: 'INVALID_SIZE_COMBINATION',
      message: 'Flyouts cannot be size "l" unless the other flyout is "fill"'
    };
  }
  return null;
}

/**
 * Creates a user-friendly error message for validation errors
 */
function createValidationErrorMessage(error) {
  console.error(error);
  var prefix = "EuiFlyout validation error";
  switch (error.type) {
    case 'INVALID_SIZE_TYPE':
    case 'INVALID_SIZE_COMBINATION':
    case 'INVALID_FLYOUT_MENU_TITLE':
      return "".concat(prefix, ": ").concat(error.message);
    default:
      return "".concat(prefix, ": Unknown validation error");
  }
}