"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateConfig = validateConfig;
exports.validateConnector = validateConnector;
exports.validateParams = validateParams;
exports.validateSecrets = validateSecrets;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _zod = require("@kbn/zod");
var _v = require("@kbn/zod/v4");
var _format_zod_error = require("./format_zod_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function validateParams(actionType, value, validatorServices) {
  return validateWithSchema(actionType, 'params', value, validatorServices);
}
function validateConfig(actionType, value, validatorServices) {
  return validateWithSchema(actionType, 'config', value, validatorServices);
}
function validateSecrets(actionType, value, validatorServices) {
  return validateWithSchema(actionType, 'secrets', value, validatorServices);
}
function validateConnector(actionType, value) {
  if (actionType.validate && actionType.validate.connector) {
    try {
      const connectorValue = value;

      // Connector config and secrets should be defined
      if (connectorValue.config == null) {
        throw new Error(`config must be defined`);
      }
      if (connectorValue.secrets == null) {
        throw new Error(`secrets must be defined`);
      }
      const result = actionType.validate.connector(connectorValue.config, connectorValue.secrets);
      if (result !== null) {
        throw new Error(result);
      }
    } catch (err) {
      throw _boom.default.badRequest(`error validating action type connector: ${err.message}`);
    }
  }
  return null;
}
function validateWithSchema(actionType, key, value, validatorServices) {
  if (actionType.validate) {
    let name;
    try {
      switch (key) {
        case 'params':
          name = 'action params';
          if (actionType.validate.params) {
            const validatedValue = actionType.validate.params.schema.parse(value);
            if (actionType.validate.params.customValidator) {
              actionType.validate.params.customValidator(validatedValue, validatorServices);
            }
            return validatedValue;
          }
          break;
        case 'config':
          name = 'connector type config';
          if (actionType.validate.config) {
            const validatedValue = actionType.validate.config.schema.parse(value);
            if (actionType.validate.config.customValidator) {
              actionType.validate.config.customValidator(validatedValue, validatorServices);
            }
            return validatedValue;
          }
          break;
        case 'secrets':
          name = 'connector type secrets';
          if (actionType.validate.secrets && value !== undefined && value !== null) {
            const validatedValue = actionType.validate.secrets.schema.parse(value);
            if (actionType.validate.secrets.customValidator) {
              actionType.validate.secrets.customValidator(validatedValue, validatorServices);
            }
            return validatedValue;
          }
          break;
        default:
          // should never happen, but left here for future-proofing
          throw new Error(`invalid actionType validate key: ${key}`);
      }
    } catch (err) {
      let errMessage = err.message;
      if (err instanceof _zod.z.ZodError) {
        errMessage = (0, _format_zod_error.formatZodError)(err);
      } else if (err instanceof _v.z.ZodError) {
        errMessage = _v.z.prettifyError(err);
      }
      // we can't really i18n this yet, since the err.message isn't i18n'd itself
      throw _boom.default.badRequest(`error validating ${name}: ${errMessage}`);
    }
  }
  return value;
}