"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.indexExplorerTool = void 0;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _onechatGenaiUtils = require("@kbn/onechat-genai-utils");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const indexExplorerSchema = _zod.z.object({
  query: _zod.z.string().describe('A natural language query to infer which indices, aliases or datastreams to use.'),
  limit: _zod.z.number().optional().describe('(optional) Limit the max number of indices to return. Defaults to 1.'),
  indexPattern: _zod.z.string().optional().describe('(optional) Index pattern to filter indices by. Defaults to *.')
});
const indexExplorerTool = () => {
  return {
    id: _onechatCommon.platformCoreTools.indexExplorer,
    type: _onechatCommon.ToolType.builtin,
    description: `List relevant indices, aliases and datastreams based on a natural language query.

The 'indexPattern' parameter can be used to filter indices by a specific pattern, e.g. 'foo*'.
This should *only* be used if you know what you're doing (e.g. if the user explicitly specified a pattern).
Otherwise, leave it empty to search against all indices.

*Example:*
User: "Show me my latest alerts"
You: call tool 'index_explorer' with { query: 'indices containing user alerts' }
Tool result: [{ type: "index", name: '.alerts' }]
`,
    schema: indexExplorerSchema,
    handler: async ({
      query: nlQuery,
      indexPattern = '*',
      limit = 1
    }, {
      esClient,
      modelProvider,
      logger
    }) => {
      logger.debug(`Index explorer tool called with query: ${nlQuery}, indexPattern: ${indexPattern}, limit: ${limit}`);
      const model = await modelProvider.getDefaultModel();
      const response = await (0, _onechatGenaiUtils.indexExplorer)({
        nlQuery,
        indexPattern,
        limit,
        esClient: esClient.asCurrentUser,
        model
      });
      return {
        results: [{
          type: _tool_result.ToolResultType.other,
          data: {
            resources: response.resources.map(resource => ({
              type: resource.type,
              name: resource.name,
              reason: resource.reason
            }))
          }
        }]
      };
    },
    tags: []
  };
};
exports.indexExplorerTool = indexExplorerTool;