"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getWindowParametersForTrigger = getWindowParametersForTrigger;
var _get_window_parameters = require("./get_window_parameters");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Calculates window parameters, adjusting the window based on a
 * change point and interval. If a change point is specified and falls within
 * the startRange, the window is adjusted around the change point. Otherwise,
 * the window is determined by the startRange and interval.
 *
 * @param startRange The start timestamp or window parameters. If a number,
 *                   it's the start timestamp; if an object, it's assumed to be
 *                   window parameters and is returned directly.
 * @param interval Interval in milliseconds for extending the window or
 *                 adjusting the start range.
 * @param timeRangeEarliest Earliest timestamp in milliseconds in the time range.
 * @param timeRangeLatest Latest timestamp in milliseconds in the time range.
 * @param changePoint Optional change point with `startTs` and `endTs`
 *                    properties. Adjusts window parameters if within `startRange`.
 * @returns Window parameters
 */
function getWindowParametersForTrigger(startRange, interval, timeRangeEarliest, timeRangeLatest, changePoint) {
  if (typeof startRange === 'number' && changePoint && startRange >= changePoint.startTs && startRange <= changePoint.endTs) {
    return (0, _get_window_parameters.getWindowParameters)(changePoint.startTs + interval, timeRangeEarliest, timeRangeLatest + interval, changePoint.endTs, interval);
  } else if (typeof startRange === 'number') {
    return (0, _get_window_parameters.getWindowParameters)(startRange + interval / 2, timeRangeEarliest, timeRangeLatest + interval);
  }
  return startRange;
}