"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findRuleTemplates = findRuleTemplates;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _esQuery = require("@kbn/es-query");
var _rule_template = require("../../../../data/rule_template");
var _saved_objects = require("../../../../saved_objects");
var _schema = require("./schema");
var _transform_raw_rule_template_to_rule_template = require("../../transforms/transform_raw_rule_template_to_rule_template");
var _filters = require("../../../../rules_client/common/filters");
var _common = require("../../../../rules_client/common");
var _authorization = require("../../../../authorization");
var _alerting_authorization_kuery = require("../../../../authorization/alerting_authorization_kuery");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function findRuleTemplates(context, params) {
  try {
    _schema.findRuleTemplatesParamsSchema.validate(params);
  } catch (error) {
    throw _boom.default.badRequest(`Error validating find template data - ${error.message}`);
  }

  // we follow the same auth patterns as in the find rules API, however the implementation is slightly
  // different because rule templates do not have consumers. If the user has access to a rule type for
  // at least one consumer, then all the templates of that rule type should be returned.
  const authorizedRuleTypes = await context.authorization.getAllAuthorizedRuleTypesFindOperation({
    authorizationEntity: _authorization.AlertingAuthorizationEntity.Rule
  });

  // user has no access to any rule types
  if (!authorizedRuleTypes.size) {
    throw _boom.default.forbidden(`Unauthorized to find ${_authorization.AlertingAuthorizationEntity.Rule}s for any rule types`);
  }

  // build the filter which applies the rule type constraints for the authorized user.
  const authorizedRuleTypeIds = Array.from(authorizedRuleTypes.keys());
  const authorizationFilter = _esQuery.nodeBuilder.or(authorizedRuleTypeIds.map(ruleTypeId => {
    (0, _alerting_authorization_kuery.ensureFieldIsSafeForQuery)('ruleTypeId', ruleTypeId);
    return _esQuery.nodeBuilder.is(`${_saved_objects.RULE_TEMPLATE_SAVED_OBJECT_TYPE}.attributes.ruleTypeId`, ruleTypeId);
  }));
  const {
    ruleTypeId,
    tags,
    perPage,
    page,
    search,
    defaultSearchOperator,
    sortField,
    sortOrder
  } = params;
  const ruleTypeFilter = ruleTypeId ? (0, _filters.buildRuleTypeIdsFilter)([ruleTypeId], _saved_objects.RULE_TEMPLATE_SAVED_OBJECT_TYPE) : undefined;
  const tagsFilter = tags ? (0, _filters.buildTagsFilter)(tags, _saved_objects.RULE_TEMPLATE_SAVED_OBJECT_TYPE) : undefined;
  const combinedFilters = (0, _filters.combineFilters)([ruleTypeFilter, tagsFilter], 'and');
  const finalFilter = (0, _filters.combineFilterWithAuthorizationFilter)(combinedFilters, authorizationFilter);

  // aside from rule type, these are the only mapped fields.
  // it doesn't make much sense to expose a param to customize this yet.
  // we should add 'description' here when it's available.
  const searchFields = ['name', 'tags'];
  const {
    page: resultPage,
    per_page: resultPerPage,
    total,
    saved_objects: data
  } = await (0, _rule_template.findRuleTemplatesSo)({
    savedObjectsClient: context.unsecuredSavedObjectsClient,
    savedObjectsFindOptions: {
      page,
      perPage,
      search,
      searchFields,
      defaultSearchOperator,
      sortField: (0, _common.mapSortField)(sortField),
      sortOrder,
      filter: finalFilter
    }
  });

  // this is a second layer of defence which validates the auth filter was applied correctly
  const ensureRuleTypeIsAuthorized = returnedRuleTypeId => {
    if (!authorizedRuleTypes.has(returnedRuleTypeId)) {
      throw _boom.default.forbidden(`Unauthorized to find ${_authorization.AlertingAuthorizationEntity.Rule} for rule type "${returnedRuleTypeId}"`);
    }
  };
  const authorizedData = data.map(so => {
    ensureRuleTypeIsAuthorized(so.attributes.ruleTypeId);
    return so;
  });
  const transformedData = authorizedData.map(so => (0, _transform_raw_rule_template_to_rule_template.transformRawRuleTemplateToRuleTemplate)({
    id: so.id,
    attributes: so.attributes
  }));
  return {
    page: resultPage,
    perPage: resultPerPage,
    total,
    data: transformedData
  };
}