"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TASK_ID = void 0;
exports.initializeApiKeyInvalidator = initializeApiKeyInvalidator;
exports.scheduleApiKeyInvalidatorTask = scheduleApiKeyInvalidatorTask;
exports.taskRunner = taskRunner;
var _server = require("@kbn/task-manager-plugin/server");
var _saved_objects = require("@kbn/actions-plugin/server/constants/saved_objects");
var _task_state = require("./task_state");
var _ = require("..");
var _saved_objects2 = require("../saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TASK_TYPE = 'alerts_invalidate_api_keys';
const TASK_ID = exports.TASK_ID = `Alerts-${TASK_TYPE}`;
function initializeApiKeyInvalidator(logger, coreStartServices, taskManager, config) {
  registerApiKeyInvalidatorTaskDefinition(logger, coreStartServices, taskManager, config);
}
async function scheduleApiKeyInvalidatorTask(logger, config, taskManager) {
  const interval = config.invalidateApiKeysTask.interval;
  try {
    await taskManager.ensureScheduled({
      id: TASK_ID,
      taskType: TASK_TYPE,
      schedule: {
        interval
      },
      state: _task_state.emptyState,
      params: {}
    });
  } catch (e) {
    logger.error(`Error scheduling ${TASK_ID} task, received ${e.message}`);
  }
}
function registerApiKeyInvalidatorTaskDefinition(logger, coreStartServices, taskManager, config) {
  taskManager.registerTaskDefinitions({
    [TASK_TYPE]: {
      title: 'Invalidate alert API Keys',
      stateSchemaByVersion: _task_state.stateSchemaByVersion,
      createTaskRunner: taskRunner(logger, coreStartServices, config)
    }
  });
}
function taskRunner(logger, coreStartServices, config) {
  return ({
    taskInstance
  }) => {
    const state = taskInstance.state;
    return {
      async run() {
        let totalInvalidated = 0;
        try {
          const [{
            savedObjects
          }, {
            encryptedSavedObjects,
            security
          }] = await coreStartServices;
          const savedObjectsClient = savedObjects.createInternalRepository([_.API_KEY_PENDING_INVALIDATION_TYPE, _saved_objects2.AD_HOC_RUN_SAVED_OBJECT_TYPE, _saved_objects.ACTION_TASK_PARAMS_SAVED_OBJECT_TYPE]);
          const encryptedSavedObjectsClient = encryptedSavedObjects.getClient({
            includedHiddenTypes: [_.API_KEY_PENDING_INVALIDATION_TYPE]
          });
          totalInvalidated = await (0, _server.runInvalidate)({
            encryptedSavedObjectsClient,
            invalidateApiKeyFn: security === null || security === void 0 ? void 0 : security.authc.apiKeys.invalidateAsInternalUser,
            logger,
            removalDelay: config.invalidateApiKeysTask.removalDelay,
            savedObjectsClient,
            savedObjectType: _.API_KEY_PENDING_INVALIDATION_TYPE,
            savedObjectTypesToQuery: [{
              type: _saved_objects2.AD_HOC_RUN_SAVED_OBJECT_TYPE,
              apiKeyAttributePath: `${_saved_objects2.AD_HOC_RUN_SAVED_OBJECT_TYPE}.attributes.apiKeyId`
            }, {
              type: _saved_objects.ACTION_TASK_PARAMS_SAVED_OBJECT_TYPE,
              apiKeyAttributePath: `${_saved_objects.ACTION_TASK_PARAMS_SAVED_OBJECT_TYPE}.attributes.apiKeyId`
            }]
          });
          const updatedState = {
            runs: (state.runs || 0) + 1,
            total_invalidated: totalInvalidated
          };
          return {
            state: updatedState,
            schedule: {
              interval: config.invalidateApiKeysTask.interval
            }
          };
        } catch (e) {
          logger.warn(`Error executing alerting apiKey invalidation task: ${e.message}`);
          const updatedState = {
            runs: state.runs + 1,
            total_invalidated: totalInvalidated
          };
          return {
            state: updatedState,
            schedule: {
              interval: config.invalidateApiKeysTask.interval
            }
          };
        }
      }
    };
  };
}