"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Gap = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _constants = require("../../../../common/constants");
var _interval_utils = require("./interval_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class Gap {
  constructor({
    ruleId,
    timestamp,
    range,
    filledIntervals = [],
    inProgressIntervals = [],
    internalFields,
    updatedAt,
    failedAutoFillAttempts
  }) {
    (0, _defineProperty2.default)(this, "_range", void 0);
    (0, _defineProperty2.default)(this, "_filledIntervals", void 0);
    (0, _defineProperty2.default)(this, "_inProgressIntervals", void 0);
    (0, _defineProperty2.default)(this, "_internalFields", void 0);
    (0, _defineProperty2.default)(this, "_timestamp", void 0);
    (0, _defineProperty2.default)(this, "_updatedAt", void 0);
    (0, _defineProperty2.default)(this, "_failedAutoFillAttempts", void 0);
    (0, _defineProperty2.default)(this, "_ruleId", void 0);
    this._range = (0, _interval_utils.normalizeInterval)(range);
    this._filledIntervals = (0, _interval_utils.mergeIntervals)(filledIntervals.map(_interval_utils.normalizeInterval));
    this._inProgressIntervals = (0, _interval_utils.mergeIntervals)(inProgressIntervals.map(_interval_utils.normalizeInterval));
    if (internalFields) {
      this._internalFields = internalFields;
    }
    if (timestamp) {
      this._timestamp = timestamp;
    }
    this._updatedAt = updatedAt !== null && updatedAt !== void 0 ? updatedAt : new Date().toISOString();
    this._ruleId = ruleId;
    this._failedAutoFillAttempts = failedAutoFillAttempts !== null && failedAutoFillAttempts !== void 0 ? failedAutoFillAttempts : 0;
  }
  fillGap(interval) {
    const clippedInterval = (0, _interval_utils.clipInterval)(interval, this.range);
    if (clippedInterval === null) return;
    const newFilledIntervals = (0, _interval_utils.mergeIntervals)([...this.filledIntervals, clippedInterval]);
    this._filledIntervals = newFilledIntervals;
    const newInProgressIntervals = (0, _interval_utils.subtractAllIntervals)(this.inProgressIntervals, newFilledIntervals);
    this._inProgressIntervals = newInProgressIntervals;
  }
  addInProgress(interval) {
    const clippedInterval = (0, _interval_utils.clipInterval)(interval, this.range);
    if (clippedInterval === null) return;
    const inProgressIntervals = (0, _interval_utils.subtractAllIntervals)([clippedInterval], this.filledIntervals);
    this._inProgressIntervals = (0, _interval_utils.mergeIntervals)([...this.inProgressIntervals, ...inProgressIntervals]);
  }
  get range() {
    return this._range;
  }
  get filledIntervals() {
    return this._filledIntervals;
  }
  get inProgressIntervals() {
    return this._inProgressIntervals;
  }
  get timestamp() {
    return this._timestamp;
  }
  get updatedAt() {
    return this._updatedAt;
  }
  setUpdatedAt(updatedAt) {
    this._updatedAt = updatedAt;
  }

  /**
   * unfilled = range - (filled + inProgress)
   */
  get unfilledIntervals() {
    const combined = (0, _interval_utils.mergeIntervals)([...this.filledIntervals, ...this.inProgressIntervals]);
    return (0, _interval_utils.subtractIntervals)(this.range, combined);
  }
  get totalGapDurationMs() {
    return (0, _interval_utils.intervalDuration)(this.range);
  }
  get filledGapDurationMs() {
    return (0, _interval_utils.sumIntervalsDuration)(this.filledIntervals);
  }
  get unfilledGapDurationMs() {
    return (0, _interval_utils.sumIntervalsDuration)(this.unfilledIntervals);
  }
  get inProgressGapDurationMs() {
    return (0, _interval_utils.sumIntervalsDuration)(this.inProgressIntervals);
  }
  get status() {
    if (this.filledGapDurationMs === 0) {
      return _constants.gapStatus.UNFILLED;
    } else if (this.unfilledGapDurationMs === 0 && this.inProgressGapDurationMs === 0) {
      return _constants.gapStatus.FILLED;
    } else {
      return _constants.gapStatus.PARTIALLY_FILLED;
    }
  }
  incrementFailedAutoFillAttempts() {
    var _this$_failedAutoFill;
    this._failedAutoFillAttempts = ((_this$_failedAutoFill = this._failedAutoFillAttempts) !== null && _this$_failedAutoFill !== void 0 ? _this$_failedAutoFill : 0) + 1;
  }
  get failedAutoFillAttempts() {
    var _this$_failedAutoFill2;
    return (_this$_failedAutoFill2 = this._failedAutoFillAttempts) !== null && _this$_failedAutoFill2 !== void 0 ? _this$_failedAutoFill2 : 0;
  }
  resetInProgressIntervals() {
    this._inProgressIntervals = [];
  }
  get internalFields() {
    return this._internalFields;
  }
  get ruleId() {
    return this._ruleId;
  }
  getState() {
    return {
      range: (0, _interval_utils.denormalizeInterval)(this.range),
      filledIntervals: this.filledIntervals.map(_interval_utils.denormalizeInterval),
      inProgressIntervals: this.inProgressIntervals.map(_interval_utils.denormalizeInterval),
      unfilledIntervals: this.unfilledIntervals.map(_interval_utils.denormalizeInterval),
      status: this.status,
      totalGapDurationMs: this.totalGapDurationMs,
      filledDurationMs: this.filledGapDurationMs,
      unfilledDurationMs: this.unfilledGapDurationMs,
      inProgressDurationMs: this.inProgressGapDurationMs
    };
  }

  /**
   * Returns the gap object for es
   */
  toObject() {
    return {
      range: (0, _interval_utils.denormalizeInterval)(this.range),
      filled_intervals: this.filledIntervals.map(_interval_utils.denormalizeInterval),
      in_progress_intervals: this.inProgressIntervals.map(_interval_utils.denormalizeInterval),
      unfilled_intervals: this.unfilledIntervals.map(_interval_utils.denormalizeInterval),
      status: this.status,
      total_gap_duration_ms: this.totalGapDurationMs,
      filled_duration_ms: this.filledGapDurationMs,
      unfilled_duration_ms: this.unfilledGapDurationMs,
      in_progress_duration_ms: this.inProgressGapDurationMs,
      updated_at: this._updatedAt,
      failed_auto_fill_attempts: this._failedAutoFillAttempts
    };
  }
}
exports.Gap = Gap;